<?php

namespace XFI\Import\Importer;

use XF\BbCode\ProcessorAction\AnalyzeUsage;
use XF\Db\DuplicateKeyException;
use XF\Db\Exception;
use XF\Db\PostgreSql\Adapter;
use XF\Entity\User;
use XF\Html\Renderer\BbCode;
use XF\Import\Data\Attachment;
use XF\Import\Data\BookmarkItem;
use XF\Import\Data\Category;
use XF\Import\Data\ConversationMaster;
use XF\Import\Data\ConversationMessage;
use XF\Import\Data\EditHistory;
use XF\Import\Data\Forum;
use XF\Import\Data\Node;
use XF\Import\Data\Post;
use XF\Import\Data\ReactionContent;
use XF\Import\Data\Thread;
use XF\Import\Data\UserField;
use XF\Import\Data\UserGroup;
use XF\Import\Data\UsernameChange;
use XF\Import\Data\Warning;
use XF\Import\DataHelper\Avatar;
use XF\Import\DataHelper\Moderator;
use XF\Import\DataHelper\Permission;
use XF\Import\DataHelper\ProfileBanner;
use XF\Import\DataHelper\Tag;
use XF\Import\Importer\AbstractForumImporter;
use XF\Import\Importer\StepPostsTrait;
use XF\Import\StepState;
use XF\Timer;
use XF\Util\Arr;

use function count, defined, in_array, intval;

class Discourse extends AbstractForumImporter
{
	use StepPostsTrait;

	/**
	 * @var int
	 */
	public const GROUP_GUEST = 0;

	/**
	 * @var int
	 */
	public const GROUP_ADMIN = 1;

	/**
	 * @var int
	 */
	public const GROUP_MOD = 2;

	/**
	 * @var int
	 */
	public const GROUP_STAFF = 3;

	/**
	 * @var int
	 */
	public const GROUP_SILENCED = 9;

	/**
	 * @var int
	 */
	public const GROUP_REG = 10;

	/**
	 * @var Adapter
	 */
	protected $sourceDb;

	public static function getListInfo(): array
	{
		return [
			'target' => 'XenForo',
			'source' => 'Discourse 3.x',
			'beta' => true,
		];
	}

	protected function getBaseConfigDefault(): array
	{
		return [
			'db' => [
				'host' => '',
				'username' => '',
				'password' => '',
				'dbname' => '',
				'port' => 5432,
			],
			'discourse_path' => '',
		];
	}

	public function renderBaseConfigOptions(array $vars): string
	{
		$vars['requiredExtensions'] = $this->getRequiredExtensions();
		$vars['config'] = $this->getBaseConfigDefault();

		return $this->app->templater()->renderTemplate(
			'admin:xfi_import_config_discourse',
			$vars
		);
	}

	public function validateBaseConfig(array &$baseConfig, array &$errors): bool
	{
		$fullConfig = array_replace_recursive(
			$this->getBaseConfigDefault(),
			$baseConfig
		);
		$missingFields = false;

		$requiredExtensions = $this->getRequiredExtensions();
		foreach ($requiredExtensions AS $extension => $installed)
		{
			if (!$installed)
			{
				$errors[] = \XF::phrase('required_php_extension_x_not_found', [
					'extension' => $extension,
				]);
			}
		}

		if ($fullConfig['db']['host'])
		{
			$validDbConnection = false;

			try
			{
				$sourceDb = new Adapter(
					$fullConfig['db'],
					true
				);
				$sourceDb->getConnection();
				$validDbConnection = true;
			}
			catch (Exception $e)
			{
				$errors[] = \XF::phrase(
					'source_database_connection_details_not_correct_x',
					['message' => $e->getMessage()]
				);
			}

			if ($validDbConnection)
			{
				try
				{
					$sourceDb->fetchOne('SELECT id FROM users LIMIT 1');
				}
				catch (Exception $e)
				{
					if ($fullConfig['db']['dbname'] === '')
					{
						$errors[] = \XF::phrase('please_enter_database_name');
					}
					else
					{
						$errors[] = \XF::phrase('table_prefix_or_database_name_is_not_correct');
					}
				}
			}
		}
		else
		{
			$missingFields = true;
		}

		if ($fullConfig['discourse_path'])
		{
			$path = rtrim($fullConfig['discourse_path'], '/\\ ');

			if (!file_exists($path) || !is_dir($path))
			{
				$errors[] = \XF::phrase(
					'directory_x_not_found_is_not_readable',
					['dir' => $path]
				);
			}
			else if (
				!file_exists("{$path}/uploads") ||
				!is_dir("{$path}/uploads")
			)
			{
				$errors[] = \XF::phrase(
					'directory_x_does_not_contain_expected_contents',
					['dir' => $path]
				);
			}

			$baseConfig['discourse_path'] = $path;
		}
		else
		{
			$missingFields = true;
		}

		if ($missingFields)
		{
			$errors[] = \XF::phrase('please_complete_required_fields');
		}

		return $errors ? false : true;
	}

	/**
	 * @return bool[]
	 */
	protected function getRequiredExtensions(): array
	{
		return [
			'GMP' => function_exists('gmp_init'),
			'PostgreSQL' => defined('PGSQL_LIBPQ_VERSION'),
			'YAML' => function_exists('yaml_parse'),
		];
	}

	protected function getStepConfigDefault(): array
	{
		return [
			'users' => [
				'merge_email' => true,
				'merge_name' => false,
				'pbkdf2_iterations' => 64000,
				'pbkdf2_algorithm' => 'sha256',
			],
			'avatars' => [
				'path' => $this->baseConfig['discourse_path'],
			],
			'profileHeaders' => [
				'path' => $this->baseConfig['discourse_path'],
			],
			'uploads' => [
				'path' => $this->baseConfig['discourse_path'],
			],
		];
	}

	public function renderStepConfigOptions(array $vars): string
	{
		$vars['stepConfig'] = $this->getStepConfigDefault();

		return $this->app->templater()->renderTemplate(
			'admin:xfi_import_step_config_discourse',
			$vars
		);
	}

	public function validateStepConfig(
		array $steps,
		array &$stepConfig,
		array &$errors
	): bool
	{
		return true;
	}

	protected function doInitializeSource()
	{
		$dbConfig = $this->baseConfig['db'];
		$this->sourceDb = new Adapter($dbConfig, true);
	}

	public function getSteps(): array
	{
		return [
			'userGroups' => [
				'title' => \XF::phrase('user_groups'),
			],
			'userFields' => [
				'title' => \XF::phrase('custom_user_fields'),
			],
			'users' => [
				'title' => \XF::phrase('users'),
				'depends' => ['userGroups', 'userFields'],
			],
			'usernameChanges' => [
				'title' => \XF::phrase('username_changes'),
				'depends' => ['users'],
			],
			'avatars' => [
				'title' => \XF::phrase('avatars'),
				'depends' => ['users'],
			],
			'profileHeaders' => [
				'title' => \XF::phrase('xfi_profile_headers'),
				'depends' => ['users'],
			],
			'ignoredUsers' => [
				'title' => \XF::phrase('xfi_ignored_users'),
				'depends' => ['users'],
			],
			'warnings' => [
				'title' => \XF::phrase('warnings'),
				'depends' => ['users'],
			],
			'privateMessages' => [
				'title' => \XF::phrase('xfi_import_private_messages'),
				'depends' => ['users'],
			],
			'categories' => [
				'title' => \XF::phrase('categories'),
				'depends' => ['userGroups'],
			],
			'moderators' => [
				'title' => \XF::phrase('moderators'),
				'depends' => ['categories'],
			],
			'watchedCategories' => [
				'title' => \XF::phrase('watched_categories'),
				'depends' => ['users', 'categories'],
			],
			'threads' => [
				'title' => \XF::phrase('xfi_topics'),
				'depends' => ['categories'],
				'force' => ['posts'],
			],
			'tags' => [
				'title' => \XF::phrase('tags'),
				'depends' => ['threads'],
			],
			'posts' => [
				'title' => \XF::phrase('posts'),
				'depends' => ['threads'],
			],
			'postRevisions' => [
				'title' => \XF::phrase('xfi_post_revisions'),
				'depends' => ['posts'],
			],
			// 'polls' => [
			// 	'title' => \XF::phrase('polls'),
			// 	'depends' => ['posts']
			// ], TODO: consider polls but bear in mind tied to posts and multiple per post supported
			'uploads' => [
				'title' => \XF::phrase('xfi_uploads'),
				'depends' => ['posts'],
			],
			'likes' => [
				'title' => \XF::phrase('likes'),
				'depends' => ['posts'],
			],
			'bookmarks' => [
				'title' => \XF::phrase('bookmarks'),
				'depends' => ['posts'],
			],
		];
	}

	public function stepUserGroups(StepState $state): StepState
	{
		$groupMap = [
			static::GROUP_GUEST => User::GROUP_GUEST,
			static::GROUP_ADMIN => User::GROUP_ADMIN,
			static::GROUP_MOD => User::GROUP_MOD,
			static::GROUP_REG => User::GROUP_REG,
		];

		$groups = $this->sourceDb->fetchAllKeyed(
			'SELECT *
				FROM groups
				ORDER BY id',
			'id'
		);
		// faux user group to retain silenced users
		$groups[static::GROUP_SILENCED] = [
			'name' => 'Silenced',
			'title' => null,
		];

		$groupedPermissions = [
			static::GROUP_SILENCED => [
				'forum' => [
					'postThread' => 'deny',
					'postReply' => 'deny',
				],
				'conversation' => [
					'start' => 'deny',
				],
				'profilePost' => [
					'post' => 'deny',
					'comment' => 'deny',
				],
			],
		];

		foreach ($groups AS $sourceGroupId => $group)
		{
			if (isset($groupMap[$sourceGroupId]))
			{
				// don't import the group, just map it to one of our defaults
				$targetGroupId = $groupMap[$sourceGroupId];
				$this->logHandler(
					UserGroup::class,
					$sourceGroupId,
					$targetGroupId
				);
				continue;
			}

			$groupData = $this->mapXfKeys($group, [
				'user_title' => 'title',
			]);
			$groupData['title'] = $group['full_name'] ?? $group['name'];
			$groupData['display_style_priority'] = 5;

			/** @var UserGroup $import */
			$import = $this->newHandler(UserGroup::class);

			if ($sourceGroupId == static::GROUP_STAFF)
			{
				$import->preventRetainIds();
			}

			$import->bulkSet($groupData);

			$permissions = $groupedPermissions[$sourceGroupId] ?? [];
			if ($permissions)
			{
				$import->setPermissions($permissions);
			}

			$import->save($sourceGroupId);
			$state->imported++;
		}

		return $state->complete();
	}

	public function stepUserFields(StepState $state): StepState
	{
		$fields = $this->sourceDb->fetchAllKeyed(
			'SELECT *
				FROM user_fields
				ORDER BY id',
			'id'
		);

		$fieldOptions = $this->sourceDb->fetchAllKeyed(
			'SELECT *
				FROM user_field_options
				ORDER BY id',
			'id'
		);
		$groupedFieldOptions = Arr::arrayGroup($fieldOptions, 'user_field_id');

		$existingFields = $this->db()->fetchPairs(
			'SELECT field_id, 1
				FROM xf_user_field
				ORDER BY field_id'
		);

		$fieldMap = [];

		foreach ($fields AS $sourceFieldId => $field)
		{
			$fieldData = $this->mapXfKeys($field, [
				'required',
				'viewable_profile' => 'show_on_profile',
			]);
			$fieldData['field_id'] = $this->convertToUniqueId(
				$field['name'],
				$existingFields
			);
			$fieldData['display_order'] = $field['position'] * 10;
			$filedData['user_editable'] = $field['editable'] ? 'yes' : 'once';

			$choices = [];
			$choiceMap = [];
			switch ($field['field_type'])
			{
				case 'confirm':
					$fieldData['field_type'] = 'checkbox';
					$choices[$fieldData['field_id']] = $field['name'];
					$choiceMap['true'] = $fieldData['field_id'];
					break;

				case 'dropdown':
					$fieldData['field_type'] = 'select';

					$sourceOptions = $groupedFieldOptions[$sourceFieldId] ?? [];
					if (!$sourceOptions)
					{
						continue 2;
					}

					$choices = [];
					$existingChoices = [];
					foreach ($sourceOptions AS $option)
					{
						$choiceValue = $option['value'];
						$choiceId = $this->convertToUniqueId(
							$choiceValue,
							$existingChoices
						);
						$choices[$choiceId] = $choiceValue;
						$choiceMap[$choiceValue] = $choiceId;
					}
					break;

				case 'text_field':
				default:
					$fieldData['field_type'] = 'textbox';
					break;
			}

			$fieldMap[$sourceFieldId] = [
				'id' => $fieldData['field_id'],
				'choiceMap' => $choiceMap,
			];

			/** @var UserField $import */
			$import = $this->newHandler(UserField::class);
			$import->bulkSet($fieldData);
			$import->setTitle($field['name'], $field['description']);

			if ($choices)
			{
				$import->setChoices($choices);
			}

			$import->save($sourceFieldId);
			$state->imported++;
		}

		$this->session->extra['userFieldMap'] = $fieldMap;

		return $state->complete();
	}

	public function getStepEndUsers(): int
	{
		return (int) $this->sourceDb->fetchOne('SELECT MAX(id) FROM users');
	}

	public function stepUsers(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$users = $this->sourceDb->fetchAllKeyed(
			"SELECT users.*,
					user_avatars.gravatar_upload_id,
					user_emails.email,
					user_options.mailing_list_mode,
					user_options.email_digests,
					user_options.notification_level_when_replying,
					user_options.allow_private_messages,
					user_options.hide_profile_and_presence,
					user_options.email_level,
					user_options.email_messages_level,
					user_options.timezone,
					user_options.enable_allowed_pm_users,
					user_profiles.location,
					user_profiles.website,
					user_profiles.bio_cooked,
					user_stats.post_count,
					user_stats.topic_count,
					user_stats.bounce_score
				FROM users
				LEFT JOIN user_avatars
					ON user_avatars.user_id = users.id
				INNER JOIN user_emails
					ON user_emails.user_id = users.id AND user_emails.primary = TRUE
				INNER JOIN user_options
					ON user_options.user_id = users.id
				INNER JOIN user_profiles
					ON user_profiles.user_id = users.id
				INNER JOIN user_stats
					ON user_stats.user_id = users.id
				WHERE users.id > \$1 AND users.id <= \$2
				ORDER BY users.id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$users)
		{
			return $state->complete();
		}

		$quotedSourceUserIds = $this->sourceDb->quote(array_keys($users));

		$groupUsers = $this->sourceDb->fetchAllKeyed(
			"SELECT *
				FROM group_users
				WHERE user_id IN ({$quotedSourceUserIds})
				ORDER BY id",
			'id'
		);
		$groupedGroupUsers = Arr::arrayGroup($groupUsers, 'user_id');

		$userFields = $this->sourceDb->fetchAllKeyed(
			"SELECT *
				FROM user_custom_fields
				WHERE user_id IN ({$quotedSourceUserIds})
				ORDER BY id",
			'id'
		);
		$groupedUserFields = Arr::arrayGroup($userFields, 'user_id');

		$this->lookup('user_group', $this->pluck($users, 'primary_group_id'));

		foreach ($users AS $sourceUserId => $user)
		{
			$state->startAfter = $sourceUserId;

			/** @var \XF\Import\Data\User $import */
			$import = $this->newHandler(\XF\Import\Data\User::class);

			$groupIds = $this->pluck(
				$groupedGroupUsers[$sourceUserId] ?? [],
				'group_id'
			);

			$silencedTill = $this->convertTimestamp($user['silenced_till']);
			if ($silencedTill > \XF::$time)
			{
				// map silenced status to faux group
				$groupIds[] = static::GROUP_SILENCED;
			}

			$userData = $this->mapXfKeys($user, [
				'username',
				'email',
				'custom_title' => 'title',
			]);

			$timezone = new \DateTimeZone($user['timezone'] ?: 'Europe/London');
			$userData['timezone'] = $this->getTimezoneFromOffset(
				$timezone->getOffset(\DateTime::createFromFormat('U', 0)) / 3600,
				true
			);

			if ($user['hide_profile_and_presence'])
			{
				$userData['visible'] = false;
			}

			$userData['user_group_id'] = User::GROUP_REG;
			if ($groupIds)
			{
				$userData['secondary_group_ids'] = array_diff(
					$this->getHelper()->mapUserGroupList($groupIds),
					[User::GROUP_REG]
				);
			}
			$userData['display_style_group_id'] = $this->lookupId(
				'user_group',
				$user['primary_group_id']
			);

			$userData['message_count'] = $user['topic_count'] + $user['post_count'];

			$userData['register_date'] = $this->convertTimestamp($user['created_at']);
			$userData['last_activity'] = $this->convertTimestamp($user['last_seen_at']);

			if ($user['email_digests'] || $user['mailing_list_mode'])
			{
				$userData['last_summary_email_date'] = $this->convertTimestamp($user['last_emailed_at']);
			}

			if ($user['gravatar_upload_id'])
			{
				$userData['gravatar'] = $user['email'];
			}

			if (!$user['active'])
			{
				$userData['user_state'] = 'email_confirm';
			}
			else if (!$user['approved'])
			{
				$userData['user_state'] = 'moderated';
			}
			else if ($user['bounce_score'] >= 4) // default bounce score threshold
			{
				$userData['user_state'] = 'bounced';
			}
			else
			{
				$userData['user_state'] = 'valid';
			}

			if (in_array(static::GROUP_STAFF, $groupIds))
			{
				$userData['is_staff'] = true;
			}

			$import->bulkSetDirect('user', $userData);

			if ($user['admin'])
			{
				$import->setDirect('user', 'is_admin', true);
				$import->setAdmin([
					'is_super_admin' => true,
					'permission_cache' => [],
				]);
			}

			$suspendedAt = $this->convertTimestamp($user['suspended_at']);
			$suspendedTill = $this->convertTimestamp($user['suspended_till']);
			if ($suspendedTill > \XF::$time)
			{
				$banHistory = $this->sourceDb->fetchRow(
					'SELECT *
						FROM user_histories
						WHERE target_user_id = $1
							AND action = 10
						ORDER BY id DESC
						LIMIT 1',
					$sourceUserId
				);
				$banUserId = $banHistory['acting_user_id'] ?? 0;
				$userReason = $banHistory['details'] ?? '';

				$import->setBan([
					'ban_user_id' => $this->lookupId('user', $banUserId),
					'ban_date' => $suspendedAt,
					'end_date' => $suspendedTill,
					'user_reason' => str_replace("\n", ' ', $userReason),
				]);
			}

			$import->setRegistrationIp($user['registration_ip_address']);

			$optionData = [];

			switch ($user['email_messages_level'])
			{
				case 2: // never
					$optionData['email_on_conversation'] = false;
					$optionData['push_on_conversation'] = false;
					break;

				case 1: // away
				case 0: // always
				default:
					$optionData['email_on_conversation'] = true;
					$optionData['push_on_conversation'] = true;
					break;
			}

			switch ($user['email_level'])
			{
				case 2: // never
					$optionData['creation_watch_state'] = 'watch_no_email';
					break;

				case 1: // away
				case 0: // always
				default:
					$optionData['creation_watch_state'] = 'watch_email';
					break;

			}

			switch ($user['notification_level_when_replying'])
			{
				case 1: // normal
				case 2: // track
					$optionData['interaction_watch_state'] = '';
					break;

				case 3: // watch
				default:
					$optionData['interaction_watch_state'] = 'watch_email';
					break;
			}

			$import->bulkSetDirect('option', $optionData);

			$privacyData = [];

			if ($user['hide_profile_and_presence'])
			{
				$privacyData['allow_view_profile'] = 'none';
			}
			else
			{
				$privacyData['allow_view_profile'] = 'everyone';
			}

			if ($user['allow_private_messages'] && !$user['enable_allowed_pm_users'])
			{
				$privacyData['allow_send_personal_conversation'] = 'members';
			}
			else
			{
				$privacyData['allow_send_personal_conversation'] = 'none';
			}

			$import->bulkSetDirect('privacy', $privacyData);

			$profileData = $this->mapXfKeys($user, [
				'website',
				'location',
			]);

			if ($user['bio_cooked'])
			{
				$profileData['about'] = $this->convertMessageToBbCode(
					null,
					$user['bio_cooked']
				);
			}

			$import->bulkSetDirect('profile', $profileData);

			$customFields = [];
			$fields = $groupedUserFields[$sourceUserId] ?? [];
			$fieldMap = $this->session->extra['userFieldMap'];
			foreach ($fields AS $field)
			{
				$sourceFieldId = intval(
					str_replace('user_field_', '', $field['name'])
				);
				$sourceFieldValue = $field['value'];

				$fieldData = $fieldMap[$sourceFieldId] ?? [];
				if (!$fieldData)
				{
					continue;
				}

				$targetFieldId = $fieldData['id'];

				$targetFieldChoiceMap = $fieldData['choiceMap'];
				$targetFieldValue = $targetFieldChoiceMap
					? ($targetFieldChoiceMap[$sourceFieldValue] ?? null)
					: $sourceFieldValue;
				if (!$targetFieldValue)
				{
					continue;
				}

				$customFields[$targetFieldId] = $targetFieldValue;
			}
			$import->setCustomFields($customFields);

			if (
				isset($user['password_algorithm']) &&
				preg_match(
					'/^\$pbkdf2-([^$]+)\$i=(\d+),l=\d+\$$/',
					$user['password_algorithm'],
					$matches
				)
			)
			{
				$pbkdf2Algorithm = $matches[1];
				$pbkdf2Iterations = $matches[2];
			}
			else
			{
				$pbkdf2Algorithm = $stepConfig['pbkdf2_iterations'];
				$pbkdf2Iterations = $stepConfig['pbkdf2_iterations'];
			}

			$import->setPasswordData('XF:Discourse', [
				'hash' => $user['password_hash'],
				'algo' => $pbkdf2Algorithm,
				'salt' => $user['salt'],
				'iterations' => $pbkdf2Iterations,
			]);
			// TODO: can we import 2FA?

			if ($this->importUser($sourceUserId, $import, $stepConfig))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndUsernameChanges(): int
	{
		return (int) $this->sourceDb->fetchOne(
			'SELECT MAX(id)
				FROM user_histories
				WHERE action = 21'
		);
	}

	public function stepUsernameChanges(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$userHistories = $this->sourceDb->fetchAllKeyed(
			"SELECT user_histories.*
				FROM user_histories
				WHERE id > \$1 AND id <= \$2
					AND action = 21
				ORDER BY id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$userHistories)
		{
			return $state->complete();
		}

		$this->lookup('user', array_merge(
			$this->pluck($userHistories, 'acting_user_id'),
			$this->pluck($userHistories, 'target_user_id')
		));

		foreach ($userHistories AS $sourceUserHistoryId => $userHistory)
		{
			$state->startAfter = $sourceUserHistoryId;

			$targetUserId = $this->lookupId('user', $userHistory['target_user_id']);
			if (!$targetUserId)
			{
				continue;
			}

			$usernameChangeData = $this->mapXfKeys($userHistory, [
				'old_username' => 'previous_value',
				'new_username' => 'new_value',
			]);
			$usernameChangeData['user_id'] = $targetUserId;
			$usernameChangeData['change_state'] = 'approved';
			$usernameChangeData['change_user_id'] = $this->lookupId(
				'user',
				$userHistory['acting_user_id']
			);
			$usernameChangeData['change_date'] = $this->convertTimestamp($userHistory['created_at']);
			$usernameChangeData['visible'] = false;

			/** @var UsernameChange $import */
			$import = $this->newHandler(UsernameChange::class);
			$import->bulkSet($usernameChangeData);

			if ($import->save($sourceUserHistoryId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndAvatars(): int
	{
		return (int) $this->sourceDb->fetchOne('SELECT MAX(id) FROM user_avatars');
	}

	public function stepAvatars(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$avatars = $this->sourceDb->fetchAllKeyed(
			"SELECT user_avatars.*, uploads.url
				FROM user_avatars
				INNER JOIN uploads
					ON uploads.id = user_avatars.custom_upload_id
				WHERE user_avatars.id > \$1 AND user_avatars.id <= \$2
				ORDER BY user_avatars.id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$avatars)
		{
			return $state->complete();
		}

		$this->lookup('user', $this->pluck($avatars, 'user_id'));

		/** @var Avatar $avatarHelper */
		$avatarHelper = $this->getDataHelper(Avatar::class);

		foreach ($avatars AS $sourceAvatarId => $avatar)
		{
			$state->startAfter = $sourceAvatarId;

			$targetUserId = $this->lookupId('user', $avatar['user_id']);
			if (!$targetUserId)
			{
				continue;
			}

			$file = "{$stepConfig['path']}/{$avatar['url']}";
			if (!file_exists($file))
			{
				continue;
			}

			$user = $this->em()->find(User::class, $targetUserId, ['Profile']);
			if (!$user)
			{
				continue;
			}

			if ($avatarHelper->setAvatarFromFile($file, $user))
			{
				$state->imported++;
			}

			$this->em()->detachEntity($user);

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndProfileHeaders(): int
	{
		return (int) $this->sourceDb->fetchOne(
			'SELECT MAX(user_id)
				FROM user_profiles
				WHERE profile_background_upload_id IS NOT NULL
					OR card_background_upload_id IS NOT NULL'
		);
	}

	public function stepProfileHeaders(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$headers = $this->sourceDb->fetchAllKeyed(
			"SELECT user_profiles.*, uploads.url
				FROM user_profiles
				INNER JOIN uploads
					ON uploads.id = COALESCE(
						user_profiles.profile_background_upload_id,
						user_profiles.card_background_upload_id
					)
				WHERE user_profiles.user_id > \$1 AND user_profiles.user_id <= \$2
					AND (
						user_profiles.profile_background_upload_id IS NOT NULL OR
						user_profiles.card_background_upload_id IS NOT NULL
					)
				ORDER BY user_profiles.user_id
				LIMIT {$limit}",
			'user_id',
			[$state->startAfter, $state->end]
		);
		if (!$headers)
		{
			return $state->complete();
		}

		$this->lookup('user', array_keys($headers));

		/** @var ProfileBanner $profileBannerHelper */
		$profileBannerHelper = $this->getDataHelper(ProfileBanner::class);

		foreach ($headers AS $sourceUserId => $header)
		{
			$state->startAfter = $sourceUserId;

			$targetUserId = $this->lookupId('user', $sourceUserId);
			if (!$targetUserId)
			{
				continue;
			}

			$file = "{$stepConfig['path']}/{$header['url']}";
			if (!file_exists($file))
			{
				continue;
			}

			$user = $this->em()->find(User::class, $targetUserId, ['Profile']);
			if (!$user)
			{
				continue;
			}

			if ($profileBannerHelper->setBannerFromFile($file, $user))
			{
				$state->imported++;
			}

			$this->em()->detachEntity($user);

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndIgnoredUsers(): int
	{
		return (int) $this->sourceDb->fetchOne('SELECT MAX(id) FROM users');
	}

	public function stepIgnoredUsers(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$users = $this->sourceDb->fetchAllKeyed(
			"SELECT users.*
				FROM users
				WHERE id > \$1 AND id <= \$2
				ORDER BY id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$users)
		{
			return $state->complete();
		}

		$quotedSourceUserIds = $this->sourceDb->quote(array_keys($users));

		$ignoredUsers = $this->sourceDb->fetchAllKeyed(
			"SELECT *
				FROM ignored_users
				WHERE user_id IN ({$quotedSourceUserIds})
				ORDER BY id",
			'id'
		);
		$groupedIgnoredUsers = Arr::arrayGroup($ignoredUsers, 'user_id');

		$userIdMap = $this->lookup('user', array_merge(
			$this->pluck($users, 'id'),
			$this->pluck($ignoredUsers, 'ignored_user_id')
		));

		/** @var \XF\Import\DataHelper\User $userHelper */
		$userHelper = $this->getDataHelper(\XF\Import\DataHelper\User::class);

		foreach ($users AS $sourceUserId => $users)
		{
			$state->startAfter = $sourceUserId;

			$targetUserId = $userIdMap[$sourceUserId] ?? null;
			if (!$targetUserId)
			{
				continue;
			}

			$ignoredUsers = $groupedIgnoredUsers[$sourceUserId] ?? [];
			if (!$ignoredUsers)
			{
				continue;
			}

			$sourceIgnoredUserIds = $this->pluck($ignoredUsers, 'ignored_user_id');
			$userHelper->importIgnored($targetUserId, $this->mapIdsFromArray(
				$sourceIgnoredUserIds,
				$userIdMap
			));

			$state->imported++;

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndWarnings(): int
	{
		return (int) $this->sourceDb->fetchOne('SELECT MAX(id) FROM user_warnings');
	}

	public function stepWarnings(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$warnings = $this->sourceDb->fetchAllKeyed(
			"SELECT user_warnings.*, users.username, topics.title
				FROM user_warnings
				INNER JOIN users
					ON users.id = user_warnings.user_id
				INNER JOIN topics
					ON topics.id = user_warnings.topic_id
				WHERE user_warnings.id > \$1 AND user_warnings.id <= \$2
				ORDER BY user_warnings.id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$warnings)
		{
			return $state->complete();
		}

		$this->lookup('user', $this->pluck($warnings, 'user_id'));

		foreach ($warnings AS $sourceWarningId => $warning)
		{
			$state->startAfter = $sourceWarningId;

			$targetUserId = $this->lookupId('user', $warning['user_id']);
			if (!$targetUserId)
			{
				continue;
			}

			$warningData = [
				'content_type' => 'user',
				'content_id' => $targetUserId,
				'content_title' => $warning['username'],
				'user_id' => $targetUserId,
				'warning_date' => $this->convertTimestamp($warning['created_at']),
				'warning_user_id' => $this->lookupId('user', $warning['created_by_id']),
				'title' => $warning['title'],
				'points' => 0,
			];

			/** @var Warning $import */
			$import = $this->newHandler(Warning::class);
			$import->bulkSet($warningData);

			if ($import->save($sourceWarningId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndPrivateMessages(): int
	{
		return (int) $this->sourceDb->fetchOne(
			"SELECT MAX(id)
				FROM topics
				WHERE archetype = 'private_message'"
		);
	}

	public function stepPrivateMessages(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$topics = $this->sourceDb->fetchAllKeyed(
			"SELECT topics.*
				FROM topics
				WHERE id > \$1 AND id <= \$2
					AND archetype = 'private_message'
				ORDER BY id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$topics)
		{
			return $state->complete();
		}

		$quotedSourceTopicIds = $this->sourceDb->quote(array_keys($topics));

		$topicUsers = $this->sourceDb->fetchAllKeyed(
			"SELECT *
				FROM topic_users
				WHERE topic_id IN ({$quotedSourceTopicIds})
				ORDER BY id",
			'id'
		);
		$groupedTopicUsers = Arr::arrayGroup($topicUsers, 'topic_id');

		$posts = $this->sourceDb->fetchAllKeyed(
			"SELECT posts.*, users.username
				FROM posts
				LEFT JOIN users
					ON users.id = posts.user_id
				WHERE posts.topic_id IN ({$quotedSourceTopicIds})
					AND posts.post_type = 1
				ORDER BY posts.id",
			'id'
		);
		$groupedPosts = Arr::arrayGroup($posts, 'topic_id');

		$this->lookup('user', array_merge(
			$this->pluck($topicUsers, 'user_id'),
			$this->pluck($posts, 'user_id')
		));

		foreach ($topics AS $sourceTopicId => $topic)
		{
			$state->startAfter = $sourceTopicId;

			$conversationData = $this->mapXfKeys($topic, ['title']);
			$conversationData['conversation_open'] = !$topic['closed'];

			/** @var ConversationMaster $import */
			$import = $this->newHandler(ConversationMaster::class);
			$import->bulkSet($conversationData);

			$topicUsers = $groupedTopicUsers[$sourceTopicId] ?? [];
			foreach ($topicUsers AS $topicUser)
			{
				$targetRecipientId = $this->lookupId('user', $topicUser['user_id']);
				if (!$targetRecipientId)
				{
					continue;
				}

				// normal/muted notification levels still imply access, so we can only map to active
				$import->addRecipient($targetRecipientId, 'active', [
					'last_read_date' => $this->convertTimestamp($topicUser['last_visited_at']),
					'is_starred' => $topicUser['bookmarked'],
				]);
			}

			$posts = $groupedPosts[$sourceTopicId] ?? [];
			$postPositionMap = [];
			foreach ($posts AS $sourcePostId => $post)
			{
				$messageData = [
					'message_date' => $this->convertTimestamp($post['created_at']),
					'user_id' => $this->lookupId('user', $post['user_id']),
					'username' => $post['username'] ?? '',
				];

				$messageData['message'] = $this->convertMessageToBbCode(
					'conversation_message',
					$post['cooked'],
					$embedMetadata
				);
				$messageData['embed_metadata'] = $embedMetadata;

				/** @var ConversationMessage $importMessage */
				$importMessage = $this->newHandler(ConversationMessage::class);
				$importMessage->bulkSet($messageData);
				$import->addMessage($sourcePostId, $importMessage);

				$sourcePosition = $post['post_number'];
				$postPositionMap[$sourcePosition] = $importMessage;
			}

			$targetTopicId = $import->save($sourceTopicId);
			if ($targetTopicId)
			{
				foreach ($postPositionMap AS $sourcePosition => $importMessage)
				{
					$this->log(
						'conversation_message_pos',
						"{$sourceTopicId}_{$sourcePosition}",
						$importMessage->message_id
					);
				}

				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function setupStepCategories()
	{
		/** @var Node $parentNode */
		$parentNode = $this->newHandler(Node::class);

		$parentNode->title = 'Imported categories';
		$parentNode->parent_node_id = 0;
		$parentNode->display_order = 0;

		$parentType = $this->newHandler(Category::class);
		$parentNode->setType('Category', $parentType);

		$parentNodeId = $this->sourceDb->fetchOne('SELECT MAX(id) FROM categories') + 10;
		$this->session->extra['importContainerParentNodeId'] = $parentNode->save($parentNodeId);
	}

	public function stepCategories(StepState $state): StepState
	{
		$categories = $this->sourceDb->fetchAllKeyed(
			'SELECT *
				FROM categories
				ORDER BY id',
			'id'
		);

		$categoryFields = $this->sourceDb->fetchAll(
			'SELECT *
				FROM category_custom_fields
				ORDER BY id'
		);
		$groupedCategoryFields = [];
		foreach ($categoryFields AS $categoryField)
		{
			$sourceCategoryId = $categoryField['category_id'];
			$categoryFieldName = $categoryField['name'];
			$categoryFieldValue = $categoryField['value'];
			$groupedCategoryFields[$sourceCategoryId][$categoryFieldName] = $categoryFieldValue;
		}

		$categoryGroups = $this->sourceDb->fetchAllKeyed(
			'SELECT *
				FROM category_groups
				ORDER BY id',
			'id'
		);
		$groupedCategoryGroups = Arr::arrayGroup($categoryGroups, 'category_id');

		$categoryTree = [];
		foreach ($categories AS $sourceCategoryId => $category)
		{
			$sourceParentCategoryId = $category['parent_category_id'] ?: 0;
			$categoryTree[$sourceParentCategoryId][$sourceCategoryId] = $category;
		}

		$this->lookup('user_group', $this->pluck($categoryGroups, 'group_id'));

		$state->imported = $this->importCategoryTree(
			$categoryTree,
			$groupedCategoryFields,
			$groupedCategoryGroups
		);

		return $state->complete();
	}

	protected function importCategoryTree(
		array &$categoryTree,
		array $groupedCategoryFields,
		array $groupedCategoryGroups,
		int $sourceParentId = 0,
		int $targetParentId = 0
	): int
	{
		if (!isset($categoryTree[$sourceParentId]))
		{
			return 0;
		}

		if ($targetParentId == 0)
		{
			$targetParentId = $this->session->extra['importContainerParentNodeId'];
		}

		$total = 0;

		$sortMap = [
			'activity' => 'last_post_date',
			'created' => 'post_date',
			'posts' => 'reply_count',
		];

		foreach ($categoryTree[$sourceParentId] AS $sourceCategoryId => $category)
		{
			$categoryFields = $groupedCategoryFields[$sourceCategoryId] ?? [];
			$categoryGroups = $groupedCategoryGroups[$sourceCategoryId] ?? [];

			$nodeData = $this->mapXfKeys($category, [
				'title' => 'name',
				'description',
			]);
			$nodeData['display_order'] = $category['position'] * 10;
			$nodeData['parent_node_id'] = $targetParentId;

			/** @var Node $import */
			$import = $this->newHandler(Node::class);
			$import->bulkSet($nodeData);

			$forumData = $this->mapxfKeys($category, [
				'discussion_count' => 'topic_count',
				'message_count' => 'post_count',
			]);

			$forumData['moderate_threads'] = $categoryFields['require_topic_approval'] ?? false;
			$forumData['moderate_replies'] = $categoryFields['require_reply_approval'] ?? false;

			$categorySortOrder = $category['sort_order'];
			$forumData['default_sort_order'] = $sortMap[$categorySortOrder] ?? 'last_post_date';
			$forumData['default_sort_direction'] = $category['sort_ascending'] ? 'asc' : 'desc';

			/** @var Forum $importForum */
			$importType = $this->newHandler(Forum::class);
			$importType->bulkSet($forumData);
			$import->setType('Forum', $importType);

			$targetCategoryId = $import->save($sourceCategoryId);
			if ($targetCategoryId)
			{
				if ($categoryGroups)
				{
					/** @var Permission $permHelper */
					$permHelper = $this->getDataHelper(Permission::class);

					// all categories with group-level permissions are private
					$permHelper->insertContentGlobalPermissions(
						'node',
						$targetCategoryId,
						['general' => ['viewNode' => 'reset']]
					);

					foreach ($categoryGroups AS $categoryGroup)
					{
						$targetGroupId = $this->lookupId('user_group', $categoryGroup['group_id']);
						if (!$targetGroupId)
						{
							continue;
						}

						$permissions = [];

						switch ($categoryGroup['permission_type'])
						{
							case 1: // see, reply, create
								$permissions['general']['viewNode'] = 'content_allow';
								$permissions['forum']['postReply'] = 'content_allow';
								$permissions['forum']['postThread'] = 'content_allow';
								break;

							case 2: // see, reply
								$permissions['general']['viewNode'] = 'content_allow';
								$permissions['forum']['postReply'] = 'content_allow';
								$permissions['forum']['postThread'] = 'reset';
								break;

							case 3: // see
								$permissions['general']['viewNode'] = 'content_allow';
								$permissions['forum']['postReply'] = 'reset';
								$permissions['forum']['postThread'] = 'reset';
								break;
						}

						if (
							$targetGroupId == User::GROUP_GUEST &&
							$categoryGroup['permission_type'] < 3
						)
						{
							// only allow registered users to post threads and replies
							$permHelper->insertContentUserGroupPermissions(
								'node',
								$targetCategoryId,
								User::GROUP_GUEST,
								['general' => ['viewNode' => 'content_allow']]
							);
							$permHelper->insertContentUserGroupPermissions(
								'node',
								$targetCategoryId,
								User::GROUP_REG,
								$permissions
							);
						}
						else
						{
							$permHelper->insertContentUserGroupPermissions(
								'node',
								$targetCategoryId,
								$targetGroupId,
								$permissions
							);
						}
					}
				}

				$total++;
				$total += $this->importCategoryTree(
					$categoryTree,
					$groupedCategoryFields,
					$groupedCategoryGroups,
					$sourceCategoryId,
					$targetCategoryId
				);
			}
		}

		return $total;
	}

	public function stepModerators(StepState $state): StepState
	{
		$superModeratorUserIds = $this->sourceDb->fetchAllColumn(
			'SELECT id
				FROM users
				WHERE moderator = TRUE
				ORDER BY id'
		);

		$categoryModeratorGroupMap = $this->sourceDb->fetchPairs(
			'SELECT id, reviewable_by_group_id
				FROM categories
				WHERE reviewable_by_group_id IS NOT NULL
				ORDER BY id'
		);
		if ($categoryModeratorGroupMap)
		{
			$sourceModeratorGroupIdsQuoted = $this->sourceDb->quote(
				array_unique($categoryModeratorGroupMap)
			);

			$groupUsers = $this->sourceDb->fetchAllKeyed(
				"SELECT *
					FROM group_users
					WHERE group_id IN ({$sourceModeratorGroupIdsQuoted})
					ORDER BY id",
				'id'
			);
		}
		else
		{
			$groupUsers = [];
		}

		$userCategoriesMap = [];
		foreach ($superModeratorUserIds AS $sourceUserId)
		{
			$userCategoriesMap[$sourceUserId][] = -1;
		}
		foreach ($groupUsers AS $groupUser)
		{
			$sourceUserId = $groupUser['user_id'];
			if (in_array($sourceUserId, $superModeratorUserIds))
			{
				// super moderators will already have permissions for all categories
				continue;
			}

			$sourceGroupId = $groupUser['group_id'];

			foreach ($categoryModeratorGroupMap AS $sourceCategoryId => $sourceModeratorGroupId)
			{
				if ($sourceGroupId == $sourceModeratorGroupId)
				{
					$userCategoriesMap[$sourceUserId][] = $sourceCategoryId;
				}
			}
		}

		$this->lookup('user', array_keys($userCategoriesMap));
		$this->lookup('node', array_keys($categoryModeratorGroupMap));

		/** @var Moderator $modHelper */
		$modHelper = $this->getDataHelper(Moderator::class);

		foreach ($userCategoriesMap AS $sourceUserId => $sourceCategoryIds)
		{
			$targetUserId = $this->lookupId('user', $sourceUserId);
			if (!$targetUserId)
			{
				continue;
			}

			$forumPermissions = [
				'inlineMod',
				'stickUnstickThread',
				'lockUnlockThread',
				'manageAnyThread',
				'deleteAnyThread',
				'threadReplyBan',
				'editAnyPost',
				'deleteAnyPost',
				'warn',
				'manageAnyTag',
				'viewDeleted',
				'viewModerated',
				'undelete',
				'approveUnapprove',
				'markSolutionAnyThread',
			];

			if (in_array(-1, $sourceCategoryIds))
			{
				// super moderator
				$permissionsGrouped = [
					'general' => [
						'viewIps' => 'allow',
						'bypassUserPrivacy' => 'allow',
						'cleanSpam' => 'allow',
						'viewWarning' => 'allow',
						'warn' => 'allow',
						'manageWarning' => 'allow',
						'editBasicProfile' => 'allow',
						'approveRejectUser' => 'allow',
						'approveUsernameChange' => 'allow',
						'banUser' => 'allow',
					],
					'conversation' => [
						'editAnyMessage' => 'allow',
						'alwaysInvite' => 'allow',
					],
					'profilePost' => [
						'inlineMod' => 'allow',
						'editAny' => 'allow',
						'deleteAny' => 'allow',
						'warn' => 'allow',
						'viewDeleted' => 'allow',
						'viewModerated' => 'allow',
						'undelete' => 'allow',
						'approveUnapprove' => 'allow',
					],
					'forum' => array_fill_keys(
						$forumPermissions,
						'allow'
					),
				];

				$modHelper->importModerator(
					$targetUserId,
					true,
					[User::GROUP_MOD],
					$permissionsGrouped
				);
			}
			else
			{
				// regular moderator
				foreach ($sourceCategoryIds AS $sourceCategoryId)
				{
					$targetCategoryId = $this->lookupId('node', $sourceCategoryId);
					if (!$targetCategoryId)
					{
						continue;
					}

					$permissionsGrouped = [
						'forum' => array_fill_keys(
							$forumPermissions,
							'content_allow'
						),
					];

					$modHelper->importContentModerator(
						$targetUserId,
						'node',
						$targetCategoryId,
						$permissionsGrouped
					);
				}

				$modHelper->importModerator(
					$targetUserId,
					false,
					[User::GROUP_MOD]
				);
			}

			$state->imported++;
		}

		return $state->complete();
	}

	public function getStepEndWatchedCategories(): int
	{
		return (int) $this->sourceDb->fetchOne('SELECT MAX(id) FROM categories');
	}

	public function stepWatchedCategories(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$timer = new Timer($maxTime);

		$categoryIds = $this->sourceDb->fetchAllColumn(
			'SELECT id
				FROM categories
				WHERE id > $1
				ORDER BY id',
			$state->startAfter
		);
		if (!$categoryIds)
		{
			return $state->complete();
		}

		$this->typeMap('node');

		/** @var \XF\Import\DataHelper\Forum $forumHelper */
		$forumHelper = $this->getDataHelper(\XF\Import\DataHelper\Forum::class);

		foreach ($categoryIds AS $sourceCategoryId)
		{
			$state->startAfter = $sourceCategoryId;

			$targetNodeId = $this->lookupId('node', $sourceCategoryId);
			if (!$targetNodeId)
			{
				continue;
			}

			$categoryWatchers = $this->sourceDb->fetchAllKeyed(
				'SELECT category_users.*, user_options.email_level
					FROM category_users
					INNER JOIN user_options
						ON user_options.user_id = category_users.user_id
					WHERE category_users.category_id = $1
						AND category_users.notification_level >= 3
					ORDER BY category_users.id',
				'id',
				$sourceCategoryId
			);

			$this->lookup('user', $this->pluck($categoryWatchers, 'user_id'));

			$watchData = [];
			foreach ($categoryWatchers AS $categoryWatcher)
			{
				$targetUserId = $this->lookupId('user', $categoryWatcher['user_id']);
				if (!$targetUserId)
				{
					continue;
				}

				$watchData[$targetUserId] = [
					'notify_on' => ($categoryWatcher['notification_level'] == 4) ? 'thread' : 'message',
					'send_alert' => true,
					'send_email' => ($categoryWatcher['email_level'] < 2), // away or always
				];
			}

			$forumHelper->importForumWatchBulk($targetNodeId, $watchData);
			$state->imported += count($watchData);

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndThreads(): int
	{
		return (int) $this->sourceDb->fetchOne(
			"SELECT MAX(id)
				FROM topics
				WHERE archetype = 'regular'"
		);
	}

	public function stepThreads(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$topics = $this->sourceDb->fetchAllKeyed(
			"SELECT topics.*,
					users.username,
					deleted_by_users.username AS deleted_by_username,
					reviewables.status as reviewable_status
				FROM topics
				LEFT JOIN users
					ON users.id = topics.user_id
				LEFT JOIN users AS deleted_by_users
					ON deleted_by_users.id = topics.deleted_by_id
				INNER JOIN posts
					ON posts.topic_id = topics.id AND posts.post_number = 1
				LEFT JOIN reviewables
					ON reviewables.type IN ('ReviewableFlaggedPost', 'ReviewableQueuedPost')
						AND reviewables.target_id = posts.id
				WHERE topics.id > \$1 AND topics.id <= \$2
					AND topics.archetype = 'regular'
				ORDER BY topics.id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$topics)
		{
			return $state->complete();
		}

		$quotedSourceTopicIds = $this->sourceDb->quote(array_keys($topics));

		$topicWatchers = $this->sourceDb->fetchAllKeyed(
			"SELECT topic_users.*, user_options.email_level
				FROM topic_users
				INNER JOIN user_options
						ON user_options.user_id = topic_users.user_id
				WHERE topic_users.topic_id IN ({$quotedSourceTopicIds})
					AND topic_users.notification_level = 3
				ORDER BY topic_users.id",
			'id'
		);
		$groupedTopicWatchers = Arr::arrayGroup($topicWatchers, 'topic_id');

		$this->lookup('node', $this->pluck($topics, 'category_id'));
		$this->lookup('user', array_merge(
			$this->pluck($topics, ['user_id', 'deleted_by_id']),
			$this->pluck($topicWatchers, 'user_id')
		));

		foreach ($topics AS $sourceTopicId => $topic)
		{
			$state->startAfter = $sourceTopicId;

			$targetNodeId = $this->lookupId('node', $topic['category_id']);
			if (!$targetNodeId)
			{
				continue;
			}

			$threadData = $this->mapXfKeys($topic, [
				'title',
				'reply_count',
				'view_count' => 'views',
			]);
			$threadData['node_id'] = $targetNodeId;
			$threadData['user_id'] = $this->lookupId('user', $topic['user_id']);
			$threadData['username'] = $topic['username'] ?? '';
			$threadData['post_date'] = $this->convertTimestamp($topic['created_at']);
			$threadData['sticky'] = $topic['pinned_at'] !== null;
			$threadData['discussion_open'] = (!$topic['closed'] && !$topic['archived']);

			if ($topic['deleted_at'])
			{
				$threadData['discussion_state'] = 'deleted';
			}
			else if ($topic['reviewable_status'] === 0)
			{
				// review pending
				$threadData['discussion_state'] = 'moderated';
			}
			else
			{
				$threadData['discussion_state'] = 'visible';
			}

			/** @var Thread $import */
			$import = $this->newHandler(Thread::class);
			$import->bulkSet($threadData);
			$import->setDeletionLogData($this->extractDeletionLogData($topic));

			$topicWatchers = $groupedTopicWatchers[$sourceTopicId] ?? [];
			foreach ($topicWatchers AS $topicWatcher)
			{
				$targetThreadWatcherId = $this->lookupId('user', $topicWatcher['user_id']);
				if (!$targetThreadWatcherId)
				{
					continue;
				}

				$import->addThreadWatcher(
					$targetThreadWatcherId,
					($topicWatcher['email_level'] < 2) // away or always
				);
			}

			if ($import->save($sourceTopicId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndTags(): int
	{
		return (int) $this->sourceDb->fetchOne(
			"SELECT MAX(id)
				FROM topics
				WHERE archetype = 'regular'"
		);
	}

	public function stepTags(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$topics = $this->sourceDb->fetchAllKeyed(
			"SELECT *
				FROM topics
				WHERE id > \$1 AND id <= \$2
					AND archetype = 'regular'
				ORDER BY id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$topics)
		{
			return $state->complete();
		}

		$quotedSourceTopicIds = $this->sourceDb->quote(array_keys($topics));

		$topicTags = $this->sourceDb->fetchAllKeyed(
			"SELECT topic_tags.*, tags.name
				FROM topic_tags
				INNER JOIN tags
					ON tags.id = topic_tags.tag_id
				WHERE topic_tags.topic_id IN ({$quotedSourceTopicIds})",
			'id'
		);
		$groupedTopicTags = Arr::arrayGroup($topicTags, 'topic_id');

		$this->lookup('thread', $this->pluck($topics, 'id'));
		$this->lookup('user', $this->pluck($topics, 'user_id'));

		/** @var Tag $tagHelper */
		$tagHelper = $this->getDataHelper(Tag::class);

		foreach ($topics AS $sourceTopicId => $topic)
		{
			$state->startAfter = $sourceTopicId;

			$targetTopicId = $this->lookupId('thread', $sourceTopicId);
			if (!$targetTopicId)
			{
				continue;
			}

			$topicTags = $groupedTopicTags[$sourceTopicId] ?? [];
			if (!$topicTags)
			{
				continue;
			}

			foreach ($topicTags AS $topicTag)
			{
				$targetTagContentId = $tagHelper->importTag(
					$topicTag['name'],
					'thread',
					$targetTopicId,
					[
						'add_user_id' => $this->lookupId('user', $topic['user_id']),
						'add_date' => $this->convertTimestamp($topicTag['created_at']),
						'content_date' => $this->convertTimestamp($topic['created_at']),
					]
				);

				if ($targetTagContentId)
				{
					$state->imported++;
				}
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	/**
	 * @param int $startAfter
	 * @param int $end
	 * @param int $threadLimit
	 */
	protected function getThreadIdsForPostsStep(
		$startAfter,
		$end,
		$threadLimit
	): array
	{
		return $this->sourceDb->fetchAllColumn(
			"SELECT id
				FROM topics
				WHERE id > \$1 AND id <= \$2
				ORDER BY id
				LIMIT {$threadLimit}",
			[$startAfter, $end]
		);
	}

	/**
	 * @param int $threadId
	 * @param string|int $startDate
	 */
	protected function getPostsForPostsStep($threadId, $startDate): array
	{
		$limit = static::$postsStepLimit;

		if ($startDate == 0)
		{
			$startDate = \DateTime::createFromFormat('U', $startDate)->format('Y-m-d H:i:s.u');
		}

		return $this->sourceDb->fetchAll(
			"SELECT posts.*,
					users.username,
					deleted_by_users.username AS deleted_by_username,
					reviewables.status as reviewable_status
				FROM posts
				LEFT JOIN users
					ON users.id = posts.user_id
				LEFT JOIN users AS deleted_by_users
					ON deleted_by_users.id = posts.deleted_by_id
				LEFT JOIN reviewables
					ON reviewables.type IN ('ReviewableFlaggedPost', 'ReviewableQueuedPost')
						AND reviewables.target_id = posts.id
				WHERE posts.topic_id = \$1
					AND posts.created_at > \$2
					AND posts.post_type = 1
				ORDER BY posts.created_at, posts.id
				LIMIT {$limit}",
			[$threadId, $startDate]
		);
	}

	protected function lookupUsers(array $posts)
	{
		$this->lookup('user', $this->pluck($posts, [
			'user_id',
			'last_editor_id',
		]));
	}

	protected function getPostDateField(): string
	{
		return 'created_at';
	}

	protected function getPostIdField(): string
	{
		return 'id';
	}

	/**
	 * @param int $newThreadId
	 */
	protected function handlePostImport(
		array $post,
		$newThreadId,
		StepState $state
	): Post
	{
		$postData = [
			'thread_id' => $newThreadId,
			'user_id' => $this->lookupId('user', $post['user_id']),
			'username' => $post['username'] ?? '',
			'post_date' => $this->convertTimestamp($post['created_at']),
			'position' => $state->extra['postPosition'],
		];

		if ($post['post_number'] == 1)
		{
			// deleted/moderated states are set at the thread level
			$postData['message_state'] = 'visible';
		}
		else
		{
			if ($post['deleted_at'])
			{
				$postData['message_state'] = 'deleted';
			}
			else if ($post['reviewable_status'] === 0)
			{
				// review pending
				$postData['message_state'] = 'moderated';
			}
			else
			{
				$postData['message_state'] = 'visible';
			}
		}

		$postData['message'] = $this->convertMessageToBbCode(
			'post',
			$post['cooked'],
			$embedMetadata
		);
		$postData['embed_metadata'] = $embedMetadata;

		if (
			$post['updated_at'] != $post['created_at'] &&
			$post['last_editor_id'] >= 0
		)
		{
			$postData['last_edit_date'] = $this->convertTimestamp($post['updated_at']);
			$postData['last_edit_user_id'] = $this->lookupId('user', $post['last_editor_id']);
			$postData['edit_count'] = $post['self_edits'];
		}

		/** @var Post $import */
		$import = $this->newHandler(Post::class);
		$import->bulkSet($postData);
		$import->setDeletionLogData($this->extractDeletionLogData($post));

		return $import;
	}

	protected function afterPostImport(Post $import, array $sourceData, int $newId)
	{
		// TODO: the new ID is not always in the import object
		$oldPositionId = $sourceData['topic_id'] . '_' . $sourceData['post_number'];
		$this->log('post_pos', $oldPositionId, $newId);
	}

	public function getStepEndPostRevisions(): int
	{
		return (int) $this->sourceDb->fetchOne('SELECT MAX(post_id) FROM post_revisions');
	}

	public function stepPostRevisions(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$sourcePostIds = $this->sourceDb->fetchAllColumn(
			"SELECT DISTINCT post_id
				FROM post_revisions
				WHERE post_id > \$1 AND post_id <= \$2
				ORDER BY post_id
				LIMIT {$limit}",
			[$state->startAfter, $state->end]
		);
		if (!$sourcePostIds)
		{
			return $state->complete();
		}

		$quotedSourcePostIds = $this->sourceDb->quote($sourcePostIds);
		$postRevisions = $this->sourceDb->fetchAllKeyed(
			"SELECT *
				FROM post_revisions
				WHERE post_id IN ({$quotedSourcePostIds})
				ORDER BY id",
			'id'
		);
		$groupedPostRevisions = Arr::arrayGroup($postRevisions, 'post_id');

		$this->lookup('post', $sourcePostIds);
		$this->lookup('user', $this->pluck($postRevisions, 'user_id'));

		foreach ($groupedPostRevisions AS $sourcePostId => $postRevisions)
		{
			$state->startAfter = $sourcePostId;

			$targetPostId = $this->lookupId('post', $sourcePostId);
			if (!$targetPostId)
			{
				continue;
			}

			foreach ($postRevisions AS $sourcePostRevisionId => $postRevision)
			{
				$modifications = yaml_parse($postRevision['modifications']);
				if (empty($modifications['cooked']))
				{
					// no changes to post content
					continue;
				}

				[$oldText, $newText] = $modifications['cooked'];
				if ($oldText == $newText)
				{
					// no changes to post content
					continue;
				}

				$editHistoryData = [
					'content_type' => 'post',
					'content_id' => $targetPostId,
					'edit_user_id' => $this->lookupId('user', $postRevision['user_id']),
					'edit_date' => $this->convertTimestamp($postRevision['created_at']),
				];

				$editHistoryData['old_text'] = $this->convertMessageToBbCode(
					'post',
					$oldText
				);

				/** @var EditHistory $import */
				$import = $this->newHandler(EditHistory::class);
				$import->bulkSet($editHistoryData);

				if ($import->save($sourcePostRevisionId))
				{
					$state->imported++;
				}
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function setupStepUploads()
	{
		$legacyUploads = (bool) $this->sourceDb->fetchOne("SELECT relname FROM pg_class WHERE relname = 'post_uploads'");
		$this->session->extra['legacyUploadsTable'] = $legacyUploads;
	}

	public function getStepEndUploads(): int
	{
		$table = $this->session->extra['legacyUploadsTable'] ? 'post_uploads' : 'upload_references';
		return (int) $this->sourceDb->fetchOne('SELECT MAX(id) FROM ' . $table);
	}

	public function stepUploads(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$legacyUploads = $this->session->extra['legacyUploadsTable'];

		if ($legacyUploads)
		{
			$postUploads = $this->sourceDb->fetchAllKeyed(
				"SELECT post_uploads.*,
					topics.archetype AS topic_archetype,
					uploads.user_id, uploads.original_filename, uploads.url, uploads.created_at, uploads.origin
				FROM post_uploads
				INNER JOIN posts
					ON posts.id = post_uploads.post_id
				INNER JOIN topics
					ON topics.id = posts.topic_id
				INNER JOIN uploads
					ON uploads.id = post_uploads.upload_id
				WHERE post_uploads.id > \$1 AND post_uploads.id <= \$2
				ORDER BY post_uploads.id
				LIMIT {$limit}",
				'id',
				[$state->startAfter, $state->end]
			);
		}
		else
		{
			$postUploads = $this->sourceDb->fetchAllKeyed(
				"SELECT refs.*,
					refs.target_id AS post_id,
					topics.archetype AS topic_archetype,
					uploads.user_id, uploads.original_filename, uploads.url, uploads.created_at, uploads.origin
				FROM upload_references AS refs
				INNER JOIN posts
					ON posts.id = refs.target_id AND refs.target_type = 'Post'
				INNER JOIN topics
					ON topics.id = posts.topic_id
				INNER JOIN uploads
					ON uploads.id = refs.upload_id
				WHERE refs.id > \$1 AND refs.id <= \$2
				ORDER BY refs.id
				LIMIT {$limit}",
				'id',
				[$state->startAfter, $state->end]
			);
		}
		if (!$postUploads)
		{
			return $state->complete();
		}

		$this->lookup('post', $this->pluck($postUploads, 'post_id'));
		$this->lookup('user', $this->pluck($postUploads, 'user_id'));

		foreach ($postUploads AS $sourcePostUploadId => $postUpload)
		{
			$state->startAfter = $sourcePostUploadId;

			if (!empty($postUpload['origin']))
			{
				// proxied image, skip
				continue;
			}

			switch ($postUpload['topic_archetype'])
			{
				case 'regular':
					$contentType = 'post';
					break;

				case 'private_message':
					$contentType = 'conversation_message';
					break;

				default:
					$contentType = null;
					break;
			}
			if (!$contentType)
			{
				continue;
			}

			$targetContentId = $this->lookupId($contentType, $postUpload['post_id']);
			if (!$targetContentId)
			{
				continue;
			}

			$file = "{$stepConfig['path']}/{$postUpload['url']}";
			if (!file_exists($file))
			{
				continue;
			}

			$attachment = [
				'content_type' => $contentType,
				'content_id' => $targetContentId,
				'attach_date' => $this->convertTimestamp($postUpload['created_at']),
				'unassociated' => false,
			];

			/** @var Attachment $import */
			$import = $this->newHandler(Attachment::class);
			$import->bulkSet($attachment);
			$import->setSourceFile($file, $postUpload['original_filename']);
			$import->setDataUserId($this->lookupId('user', $postUpload['user_id']));
			$import->setDataExtra(
				'upload_date',
				$this->convertTimestamp($postUpload['created_at'])
			);
			$import->setContainerCallback([$this, 'rewriteEmbeddedAttachments']);

			if ($import->save($postUpload['upload_id']))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndLikes(): int
	{
		return (int) $this->sourceDb->fetchOne(
			'SELECT MAX(id)
				FROM post_actions
				WHERE post_action_type_id = 2'
		);
	}

	public function stepLikes(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$likes = $this->sourceDb->fetchAllKeyed(
			"SELECT post_actions.*,
					posts.user_id AS post_user_id,
					topics.archetype AS topic_archetype
				FROM post_actions
				INNER JOIN posts
					ON posts.id = post_actions.post_id
				INNER JOIN topics
					ON topics.id = posts.topic_id
				WHERE post_actions.id > \$1 AND post_actions.id <= \$2
					AND post_actions.post_action_type_id = 2
				ORDER BY post_actions.id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$likes)
		{
			return $state->complete();
		}

		$this->lookup('post', $this->pluck($likes, 'post_id'));
		$this->lookup('conversation_message', $this->pluck($likes, 'post_id'));
		$this->lookup('user', $this->pluck($likes, ['user_id', 'post_user_id']));

		foreach ($likes AS $sourceLikeId => $like)
		{
			$state->startAfter = $sourceLikeId;

			switch ($like['topic_archetype'])
			{
				case 'regular':
					$contentType = 'post';
					break;

				case 'private_message':
					$contentType = 'conversation_message';
					break;

				default:
					$contentType = null;
					break;
			}
			if (!$contentType)
			{
				continue;
			}

			$targetContentId = $this->lookupId($contentType, $like['post_id']);
			if (!$targetContentId)
			{
				continue;
			}

			$targetReactionUserId = $this->lookupId('user', $like['user_id']);
			if (!$targetReactionUserId)
			{
				continue;
			}

			/** @var ReactionContent $import */
			$import = $this->newHandler(ReactionContent::class);
			$import->setReactionId(1);
			$import->bulkSet([
				'content_type' => $contentType,
				'content_id' => $targetContentId,
				'reaction_user_id' => $targetReactionUserId,
				'reaction_date' => $this->convertTimestamp($like['created_at']),
				'content_user_id' => $this->lookupId('user', $like['post_user_id']),
			]);

			if ($import->save($sourceLikeId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	public function getStepEndBookmarks(): int
	{
		return (int) $this->sourceDb->fetchOne('SELECT MAX(id) FROM bookmarks');
	}

	public function stepBookmarks(
		StepState $state,
		array $stepConfig,
		int $maxTime
	): StepState
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$bookmarks = $this->sourceDb->fetchAllKeyed(
			"SELECT bookmarks.*
				FROM bookmarks
				WHERE bookmarks.id > \$1 AND bookmarks.id <= \$2
					AND bookmarks.bookmarkable_type IN ('Post', 'Topic')
				ORDER BY bookmarks.id
				LIMIT {$limit}",
			'id',
			[$state->startAfter, $state->end]
		);
		if (!$bookmarks)
		{
			return $state->complete();
		}

		$bookmarksGrouped = Arr::arrayGroup($bookmarks, 'bookmarkable_type');

		if (!empty($bookmarksGrouped['Topic']))
		{
			$quotedTopicIds = $this->sourceDb->quote(
				$this->pluck($bookmarksGrouped['Topic'], 'bookmarkable_id')
			);
			$postMap = $this->sourceDb->fetchPairs(
				"SELECT posts.topic_id, posts.id
					FROM posts
					WHERE topic_id IN ({$quotedTopicIds})
					AND post_number = 1"
			);
		}
		else
		{
			$postMap = [];
		}

		if (!empty($bookmarksGrouped['Post']))
		{
			$postIds = array_merge(
				$this->pluck($bookmarksGrouped['Post'], 'bookmarkable_id'),
				$postMap
			);
		}
		else
		{
			$postIds = $postMap;
		}

		$this->lookup('user', $this->pluck($bookmarks, 'user_id'));
		$this->lookup('post', $postIds);

		foreach ($bookmarks AS $sourceBookmarkId => $bookmark)
		{
			$state->startAfter = $sourceBookmarkId;

			$targetUserId = $this->lookupId('user', $bookmark['user_id']);
			if (!$targetUserId)
			{
				continue;
			}

			switch ($bookmark['bookmarkable_type'])
			{
				case 'Post':
					$sourcePostId = $bookmark['bookmarkable_id'];
					break;

				case 'Topic':
					$sourcePostId = $postMap[$bookmark['bookmarkable_id']];
					break;

				default:
					$sourcePostId = null;
					break;
			}
			if (!$sourcePostId)
			{
				continue;
			}

			$targetPostId = $this->lookupId('post', $sourcePostId);
			if (!$targetPostId)
			{
				continue;
			}

			/** @var BookmarkItem $import */
			$import = $this->newHandler(BookmarkItem::class);
			$import->bulkSet([
				'user_id' => $targetUserId,
				'content_type' => 'post',
				'content_id' => $targetPostId,
				'bookmark_date' => $this->convertTimestamp($bookmark['created_at']),
				'message' => $bookmark['name'],
			]);

			try
			{
				if ($import->save($sourceBookmarkId))
				{
					$state->imported++;
				}
			}
			catch (DuplicateKeyException $e)
			{
				// this can happen if a user has bookmarked both a topic and the
				// first post of the topic
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	/**
	 * @param string|null $timestamp
	 */
	protected function convertTimestamp($timestamp): int
	{
		if ($timestamp === null)
		{
			return 0;
		}

		return strtotime($timestamp);
	}

	/**
	 * @param string|null $contentType
	 */
	protected function convertMessageToBbCode(
		$contentType,
		string $message,
		&$embedMetadata = []
	): string
	{
		$message = $this->rewriteContent($contentType, $message);

		$bbCodeContainer = $this->app->bbCode();

		$processor = $bbCodeContainer->processor();

		$processor->addProcessorAction(
			'usage',
			$bbCodeContainer->processorAction('usage')
		);
		$processor->addProcessorAction(
			'mentions',
			$bbCodeContainer->processorAction('mentions')
		);
		$processor->addProcessorAction(
			'autolink',
			$bbCodeContainer->processorAction('autolink')
		);
		$processor->addProcessorAction(
			'shortToEmoji',
			$bbCodeContainer->processorAction('shortToEmoji')
		);

		$message = $processor->render(
			BbCode::renderFromHtml($message),
			$bbCodeContainer->parser(),
			$bbCodeContainer->rules('import:discourse')
		);

		/** @var AnalyzeUsage $usage */
		$usage = $processor->getAnalyzer('usage');
		$embedMetadata = [
			'attachments' => $usage->getAttachments(),
			'unfurls' => $usage->getUnfurls(),
		];

		return $message;
	}

	/**
	 * @param string|null $contentType
	 */
	protected function rewriteContent($contentType, string $message): string
	{
		$message = $this->rewriteQuotes($contentType, $message);
		$message = $this->rewriteAttachments($message);

		// convert relative URIs to absolute URIs
		$message = preg_replace_callback(
			'/(?P<open><a [^>]*href=")(?P<uri>\/[^"]*)(?P<close>"[^>]*>)/iU',
			function (array $match)
			{
				$uri = $this->app->request()->convertToAbsoluteUri(
					$match['uri'],
					$this->app->options()->boardUrl
				);
				return $match['open'] . $uri . $match['close'];
			},
			$message
		);

		$rewriteMap = [
			'/<div [^>]*class="[^"]*onebox[^"]*"[^>]*>\s*<a [^>]*href="([^"]+)"[^>]*>\s*<img [^>]*>\s*<\/a>\s*<\/div>/iU' => '<p>$1</p>', // media unfurl
			'/<aside [^>]*class="[^"]*onebox[^"]*"[^>]*>.*<article [^>]*class="[^"]*onebox-body[^"]*"[^>]*>.*<a [^>]*href="([^"]+)"[^>]*>.*<\/article>.*<\/aside>/isU' => '<p>$1</p>', // link unfurl
			'/<img [^>]*class="[^"]*emoji[^"]*"[^>]*alt="(:[^"]+:)"[^>]*>/iU' => '$1', // emojis
			'/<a [^>]*class="[^"]*mention[^"]*"[^>]*>(.+)<\/a>/isU' => '$1', // mentions
			'/(<\/p>)(\s*<p>)/iU' => '$1<br>$2', // line breaks
		];

		return preg_replace(
			array_keys($rewriteMap),
			array_values($rewriteMap),
			$message
		);
	}

	/**
	 * @param string|null $contentType
	 */
	protected function rewriteQuotes($contentType, string $message): string
	{
		if (strpos($message, 'data-username') === false)
		{
			return $message;
		}

		return preg_replace_callback(
			'/<aside [^>]*class="[^"]*quote[^"]*"[^>]*data-username="(?P<username>[^"]+)"([^>]*data-post="(?P<post_id>\d+)"[^>]*data-topic="(?P<topic_id>\d+)")?[^>]*>.*<blockquote>(?P<quote>.+)<\/blockquote>.*<\/aside>/isU',
			function (array $match) use ($contentType)
			{
				$username = trim($match['username']);
				$postId = $match['post_id'] ?? 0;
				$topicId = $match['topic_id'] ?? 0;
				$quote = trim($match['quote']);

				if ($contentType && $topicId && $postId)
				{
					$contentId = $this->lookupId(
						"{$contentType}_pos",
						"{$topicId}_{$postId}",
						0
					);
				}
				else
				{
					$contentId = 0;
				}

				if (!$contentId)
				{
					return sprintf(
						'[QUOTE="%s"]%s[/QUOTE]',
						$username,
						$quote
					);
				}

				return sprintf(
					'[QUOTE="%s, %s: %d"]%s[/QUOTE]',
					$username,
					$contentType,
					$contentId,
					$quote
				);
			},
			$message
		);
	}

	protected function rewriteAttachments(string $message): string
	{
		if (strpos($message, 'data-base62-sha1') === false)
		{
			return $message;
		}

		if (strpos($message, 'lightbox-wrapper') !== false)
		{
			// remove lightbox wrapper
			$message = preg_replace(
				'/<div [^>]*class="[^"]*lightbox-wrapper[^"]*"[^>]*>\s*<a [^>]*class="[^"]*lightbox[^"]*"[^>]*>\s*(<img [^>]*>).*<\/a>\s*<\/div>/isU',
				'$1',
				$message
			);
		}

		if (!preg_match_all(
			'/<img [^>]*data-base62-sha1="(\w+)"[^>]*>/iU',
			$message,
			$matches
		))
		{
			return $message;
		}

		$encodedSha1s = $matches[1];
		$decodedSha1s = array_map(function ($string)
		{
			// the discourse base62 scheme swaps uppercase and lowercase characters
			// (when compared with the gmp base62 scheme)
			$lowercase = range('a', 'z');
			$uppercase = range('A', 'Z');
			$characterMap = array_merge(
				array_combine($lowercase, $uppercase),
				array_combine($uppercase, $lowercase)
			);
			$string = strtr($string, $characterMap);

			return gmp_strval(gmp_init($string, 62), 16);
		}, $encodedSha1s);
		$sha1Map = array_combine($encodedSha1s, $decodedSha1s);

		$quotedSha1s = $this->sourceDb->quote($decodedSha1s);
		$uploadMap = $this->sourceDb->fetchAllKeyed(
			"SELECT *
				FROM uploads
				WHERE sha1 IN ({$quotedSha1s})
				ORDER BY id",
			'sha1'
		);

		foreach ($sha1Map AS $encodedSha1 => $decodedSha1)
		{
			$message = preg_replace_callback(
				'/<img [^>]*alt="(?P<alt>[^"]*)"[^>]*data-base62-sha1="' . $encodedSha1 . '"[^>]*width="(?P<width>\d*)"[^>]*height="(?P<height>\d*)"[^>]*>/iU',
				function (array $match) use ($uploadMap, $decodedSha1)
				{
					$upload = $uploadMap[$decodedSha1] ?? [];
					if (!$upload)
					{
						// no matching upload, remove attachment
						return '';
					}

					$options = [];
					if ($match['alt'])
					{
						$options['alt'] = $match['alt'];
					}
					if ($match['width'])
					{
						$options['width'] = $match['width'] . 'px';
					}
					if ($match['height'])
					{
						$options['height'] = $match['height'] . 'px';
					}

					$options = array_map(
						function ($key, $value)
						{
							return "{$key}=\"{$value}\"";
						},
						array_keys($options),
						$options
					);
					$options = implode(' ', $options);

					if ($upload['origin'])
					{
						// proxied image, replace with image tag
						return sprintf(
							'<p>[IMG %s]%s[/IMG]</p>',
							$options,
							$upload['origin']
						);
					}

					return sprintf(
						'<p>[ATTACH type="full" %s]%d[/ATTACH]</p>',
						$options,
						$upload['id']
					);
				},
				$message
			);
		}

		return $message;
	}

	protected function extractDeletionLogData(array $record): array
	{
		return [
			'date' => $this->convertTimestamp($record['deleted_at']),
			'user_id' => $this->lookupId('user', $record['deleted_by_id'], 0),
			'username' => $record['deleted_by_username'] ?? '',
		];
	}
}
