<?php

namespace XFI\Import\Importer;

use XF\BbCode\ProcessorAction\AutoLink;
use XF\Db\Exception;
use XF\Db\Mysqli\Adapter;
use XF\Entity\User;
use XF\Import\Data\Attachment;
use XF\Import\Data\Category;
use XF\Import\Data\ConversationMaster;
use XF\Import\Data\ConversationMessage;
use XF\Import\Data\EntityEmulator;
use XF\Import\Data\Forum;
use XF\Import\Data\Node;
use XF\Import\Data\Poll;
use XF\Import\Data\PollResponse;
use XF\Import\Data\Post;
use XF\Import\Data\Thread;
use XF\Import\Data\UserField;
use XF\Import\Data\UserGroup;
use XF\Import\DataHelper\Moderator;
use XF\Import\DataHelper\Permission;
use XF\Import\Importer\AbstractForumImporter;
use XF\Import\Importer\StepPostsTrait;
use XF\Import\StepState;
use XF\Timer;
use XF\Util\Arr;
use XF\Util\File;

use function array_key_exists, count, in_array;

class MyBb extends AbstractForumImporter
{
	use StepPostsTrait;

	/**
	 * @var Adapter
	 */
	protected $sourceDb;

	public static function getListInfo()
	{
		return [
			'target' => 'XenForo',
			'source' => 'MyBB 1.8',
		];
	}

	protected function getBaseConfigDefault()
	{
		return [
			'db' => [
				'host' => '',
				'username' => '',
				'password' => '',
				'dbname' => '',
				'port' => 3306,
				'tablePrefix'   => '',
				'charset'  => '', // used for the DB connection
			],
			'charset' => '',
			'uploads_path' => '',
		];
	}

	public function renderBaseConfigOptions(array $vars)
	{
		$vars['db'] = [
			'host' => $this->app->config['db']['host'],
			'port' => $this->app->config['db']['port'],
			'username' => $this->app->config['db']['username'],
			'tablePrefix' => 'mybb_',
		];
		return $this->app->templater()->renderTemplate('admin:xfi_import_config_mybb', $vars);
	}

	public function validateBaseConfig(array &$baseConfig, array &$errors)
	{
		$baseConfig['db']['tablePrefix'] = preg_replace('/[^a-z0-9_]/i', '', $baseConfig['db']['tablePrefix']);

		$fullConfig = array_replace_recursive($this->getBaseConfigDefault(), $baseConfig);
		$missingFields = false;

		if ($fullConfig['db']['host'])
		{
			$validDbConnection = false;

			try
			{
				$sourceDb = new Adapter($fullConfig['db'], false);
				$sourceDb->getConnection();
				$validDbConnection = true;
			}
			catch (Exception $e)
			{
				$errors[] = \XF::phrase('source_database_connection_details_not_correct_x', ['message' => $e->getMessage()]);
			}

			if ($validDbConnection)
			{
				try
				{
					$sourceDb->fetchOne("
						SELECT uid
						FROM users
						LIMIT 1
					");
				}
				catch (Exception $e)
				{
					if ($fullConfig['db']['dbname'] === '')
					{
						$errors[] = \XF::phrase('please_enter_database_name');
					}
					else
					{
						$errors[] = \XF::phrase('table_prefix_or_database_name_is_not_correct');
					}
				}
			}
			else
			{
				$missingFields = true;
			}
		}

		if ($fullConfig['uploads_path'])
		{
			$path = rtrim($fullConfig['uploads_path'], '/\\ ');

			if (!file_exists($path) || !is_dir($path))
			{
				$errors[] = \XF::phrase('directory_x_not_found_is_not_readable', ['dir' => $path]);
			}
			else if (!file_exists("$path/avatars") || !is_dir("$path/avatars"))
			{
				$errors[] = \XF::phrase('directory_x_does_not_contain_expected_contents', ['dir' => $path]);
			}

			$baseConfig['uploads_path'] = $path;
		}
		else
		{
			$missingFields = true;
		}

		if ($missingFields)
		{
			$errors[] = \XF::phrase('please_complete_required_fields');
		}

		return $errors ? false : true;
	}

	protected function getStepConfigDefault()
	{
		return [
			'users' => [
				'merge_email' => true,
				'merge_name'  => false,
				'super_admins' => [],
			],
		];
	}

	public function renderStepConfigOptions(array $vars)
	{
		$vars['stepConfig'] = $this->getStepConfigDefault();
		return $this->app->templater()->renderTemplate('admin:xfi_import_step_config_mybb', $vars);
	}

	public function validateStepConfig(array $steps, array &$stepConfig, array &$errors)
	{
		foreach (array_keys($stepConfig) AS $step)
		{
			$callback = 'validateStepConfig' . ucfirst($step);

			if (method_exists($this, $callback))
			{
				$this->$callback($stepConfig[$step], $stepConfig, $errors);
			}
		}

		return $errors ? false : true;
	}

	/**
	 * Process super admin IDs
	 *
	 * @param array $config
	 * @param array $stepConfig
	 * @param array $errors
	 */
	protected function validateStepConfigUsers(array &$config, array &$stepConfig, array &$errors)
	{
		if (isset($config['super_admins']))
		{
			$config['super_admins'] = Arr::stringToArray($config['super_admins'], '/\s*,\s*/s');
		}
		else
		{
			$config['super_admins'] = [];
		}
	}

	public function getSteps()
	{
		// TODO: Additional steps to consider: importing reputation as reactions
		// TODO: importing thread prefixes
		// TODO: importing warnings / bans
		// TODO: group promotions
		// TODO: help pages

		return [
			'userGroups' => [
				'title' => \XF::phrase('user_groups'),
			],
			'userFields' => [
				'title' => \XF::phrase('custom_user_fields'),
			],
			'users' => [
				'title' => \XF::phrase('users'),
				'depends' => ['userGroups', 'userFields'],
			],
			'privateMessages' => [
				'title' => \XF::phrase('xfi_import_private_messages'),
				'depends' => ['users'],
			],
			'forums' => [
				'title' => \XF::phrase('forums'),
			],
			'moderators' => [
				'title' => \XF::phrase('moderators'),
				'depends' => ['forums', 'users'],
			],
			'threads' => [
				'title' => \XF::phrase('threads'),
				'depends' => ['forums', 'users'],
				'force' => ['posts'],
			],
			'posts' => [
				'title' => \XF::phrase('posts'),
				'depends' => ['threads'],
			],
			'polls' => [
				'title' => \XF::phrase('thread_polls'),
				'depends' => ['posts'],
			],
			'attachments' => [
				'title' => \XF::phrase('attachments'),
				'depends' => ['posts'], // can come from conversations as well, though not required
			],
		];
	}

	protected function doInitializeSource()
	{
		$dbConfig = $this->baseConfig['db'];

		$this->sourceDb = new Adapter($dbConfig, false);

		$this->dataManager->setSourceCharset($this->baseConfig['charset'], true);
	}

	// ############################## STEP: USER GROUPS #########################

	public function stepUserGroups(StepState $state, array $stepConfig)
	{
		$groups = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM usergroups
			ORDER BY gid
		", 'gid');

		foreach ($groups AS $oldId => $group)
		{
			$permissions = $this->calculateGroupPerms($group);
			$titlePriority = 5;

			$groupMap = [
				1 => User::GROUP_GUEST,

				2 => User::GROUP_REG,
				5 => User::GROUP_REG,

				3 => User::GROUP_MOD,
				6 => User::GROUP_MOD,

				4 => User::GROUP_ADMIN,
			];

			if (array_key_exists($oldId, $groupMap))
			{
				// don't import the group, just map it to one of our defaults
				$this->logHandler(UserGroup::class, $oldId, $groupMap[$oldId]);
			}
			else
			{
				$data = [
					'title' => $group['title'],
					'user_title' => $group['usertitle'],
				];

				if ($oldId == 3) // super mods
				{
					$titlePriority = 910;
				}

				/** @var UserGroup $import */
				$import = $this->newHandler(UserGroup::class);
				$import->bulkSet($data);
				$import->set('display_style_priority', $titlePriority);
				$import->setPermissions($permissions);
				$import->save($oldId);
			}

			$state->imported++;
		}

		return $state->complete();
	}

	protected function calculateGroupPerms(array $group)
	{
		$perms = [];

		if ($group['canview'])
		{
			$perms['general']['view'] = 'allow';
			$perms['general']['viewNode'] = 'allow';
			$perms['general']['editProfile'] = 'allow';
			$perms['general']['submitWithoutApproval'] = 'allow';
		}
		if ($group['canviewthreads'])
		{
			$perms['forum']['viewOthers'] = 'allow';
			$perms['forum']['viewContent'] = 'allow';
		}
		if ($group['canviewprofiles'])
		{
			$perms['general']['viewProfile'] = 'allow';
			$perms['general']['viewMemberList'] = 'allow';
			$perms['profilePost']['view'] = 'allow';
		}
		if ($group['cansearch'])
		{
			$perms['general']['search'] = 'allow';
		}
		if ($group['canuploadavatars'])
		{
			$perms['avatar']['allowed'] = 'allow';
			$perms['avatar']['maxFileSize'] = -1;
		}
		if ($group['cancustomtitle'])
		{
			$perms['general']['editCustomTitle'] = 'allow';
		}
		if (!isset($group['canusesig']) || $group['canusesig'])
		{
			$perms['general']['editSignature'] = 'allow';
		}

		if ($group['candlattachments'])
		{
			$perms['forum']['viewAttachment'] = 'allow';
		}
		if ($group['canpostattachments'])
		{
			$perms['forum']['uploadAttachment'] = 'allow';
		}

		if ($group['canpostthreads'])
		{
			$perms['forum']['postThread'] = 'allow';
			$perms['forum']['editOwnThreadTitle'] = 'allow';
		}
		if ($group['canpostreplys'])
		{
			$perms['forum']['postReply'] = 'allow';
			$perms['forum']['like'] = 'allow';
			$perms['general']['maxTaggedUsers'] = 5;
			$perms['general']['report'] = 'allow';

			$perms['profilePost']['post'] = 'allow';
			$perms['profilePost']['comment'] = 'allow';
			$perms['profilePost']['manageOwn'] = 'allow';
			$perms['profilePost']['like'] = 'allow';
		}
		if ($group['caneditposts'])
		{
			$perms['forum']['editOwnPost'] = 'allow';
			$perms['forum']['editOwnPostTimeLimit'] = -1;
			$perms['profilePost']['editOwn'] = 'allow';
		}
		if ($group['candeleteposts'])
		{
			$perms['forum']['deleteOwnPost'] = 'allow';
			$perms['profilePost']['deleteOwn'] = 'allow';
		}
		if ($group['candeletethreads'])
		{
			$perms['forum']['deleteOwnThread'] = 'allow';
		}

		if ($group['canvotepolls'])
		{
			$perms['forum']['votePoll'] = 'allow';
		}

		if ($group['cansendpms'])
		{
			$perms['conversation']['start'] = 'allow';
			$perms['conversation']['receive'] = 'allow';

			if ($group['maxpmrecipients'])
			{
				$perms['conversation']['maxRecipients'] = $group['maxpmrecipients'];
				if ($perms['conversation']['maxRecipients'] > 2147483647)
				{
					$perms['conversation']['maxRecipients'] = -1;
				}
			}
			else
			{
				$perms['conversation']['maxRecipients'] = -1;
			}
		}

		return $perms;
	}

	// ############################## STEP: USER FIELDS #########################

	public function stepUserFields(StepState $state, array $stepConfig)
	{
		$fields = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM profilefields
			WHERE fid > 2
			ORDER BY fid
		", 'fid');

		$choiceLookUps = [];
		$existingFields = $this->db()->fetchPairs("SELECT field_id, field_id FROM xf_user_field");

		foreach ($fields AS $oldId => $field)
		{
			$fieldId = $this->convertToUniqueId($field['name'], $existingFields, 25);

			$data = [
				'field_id' => $fieldId,
				'display_order' => $field['disporder'],
				'max_length' => $field['maxlength'],
				'viewable_profile' => ($field['viewableby'] == -1),
				'user_editable' => ($field['editableby'] == -1 ? 'yes' : 'never'),
				'viewable_message' => 0,
				'show_registration' => 0,
				'required' => $field['required'],
			];

			$typeParts = Arr::stringToArray($field['type'], '/\r?\n/');
			$type = array_shift($typeParts);
			$typeParts = array_unique($typeParts);

			switch ($type)
			{
				case 'textarea':
					$data['field_type'] = 'textarea';
					break;

				case 'select':
				case 'radio':
				case 'checkbox':
				case 'multiselect':
					$data['field_type'] = $type;
					$data['field_choices'] = $typeParts;
					$choiceLookUps[$oldId] = array_flip($typeParts);
					break;

				default:
					$data['field_type'] = 'textbox';
					break;
			}

			/** @var UserField $import */
			$import = $this->newHandler(UserField::class);
			$import->setTitle($field['name'], $field['description']);
			$import->bulkSet($data);

			$import->save($oldId);

			$state->imported++;
		}

		$this->session->extra['profileFieldChoices'] = $choiceLookUps;

		return $state->complete();
	}

	// ############################## STEP: USERS #############################

	public function getStepEndUsers()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(uid) FROM users") ?: 0;
	}

	public function stepUsers(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$users = $this->sourceDb->fetchAllKeyed("
			SELECT users.*, userfields.*,
				banned.dateline AS ban_dateline, banned.lifted AS ban_lifted, banned.reason AS ban_reason,
				adminoptions.permissions AS admin_permissions
			FROM users AS users
			LEFT JOIN adminoptions AS adminoptions ON
				(adminoptions.uid = users.uid)
			LEFT JOIN userfields AS userfields ON
				(userfields.ufid = users.uid)
			LEFT JOIN banned AS banned ON
				(banned.uid = users.uid AND (banned.lifted = 0 OR banned.lifted > ?))
			WHERE users.uid > ? AND users.uid <= ?
			ORDER BY users.uid
			LIMIT $limit
		", 'uid', [\XF::$time, $state->startAfter, $state->end]);

		if (!$users)
		{
			return $state->complete();
		}

		foreach ($users AS $oldId => $user)
		{
			$state->startAfter = $oldId;

			$import = $this->setupImportUser($user, $state, $stepConfig);
			if ($this->importUser($oldId, $import, $stepConfig))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	protected function setupImportUser(array $user, StepState $state, array $stepConfig)
	{
		/** @var \XF\Import\Data\User $import */
		$import = $this->newHandler(\XF\Import\Data\User::class);

		$userData = $this->mapXfKeys($user, [
			'username',
			'email',
			'last_activity' => 'lastvisit',
			'register_date' => 'regdate',
			'message_count' => 'postnum',
		]);

		$import->bulkSetDirect('user', $userData);

		if (!isset($this->session->extra['userActivationSetting']))
		{
			$this->session->extra['userActivationSetting'] = $this->sourceDb->fetchOne("
				SELECT value
				FROM settings
				WHERE name = 'regtype'
			");
		}

		if ($user['usergroup'] == 5)
		{
			$import->user_state = ($this->session->extra['userActivationSetting'] == 'admin' ? 'moderated' : 'email_confirm');
		}
		else
		{
			$import->user_state = 'valid';
		}

		// user groups
		$import->user_group_id = $this->lookupId('user_group', $user['usergroup'], User::GROUP_REG);
		if ($user['additionalgroups'])
		{
			$secondaryGroups = Arr::stringToArray($user['additionalgroups'], '/,\s*/');
			$import->secondary_group_ids = $this->lookup('user_group', $secondaryGroups);
		}

		$import->is_admin = ($user['admin_permissions'] ? 1 : 0);
		$import->is_banned = ($user['ban_dateline'] ? 1 : 0);

		if ($import->is_banned)
		{
			$import->setBan([
				'ban_user_id' => 0,
				'ban_date' => $user['ban_dateline'],
				'end_date' => $user['ban_lifted'],
				'user_reason' => $user['ban_reason'],
			]);
		}

		$import->timezone = $this->getTimezoneFromOffset($user['timezone'], $user['dstcorrection']);

		$import->visible = !$user['invisible'];

		$import->setPasswordData('XF:MyBb', [
			'hash' => $user['password'],
			'salt' => $user['salt'],
		]);

		$import->setRegistrationIp($user['regip']);

		$import->website = $user['website'];
		$import->signature = $this->convertContentToBbCode($user['signature']);

		if ($user['birthday'])
		{
			$parts = explode('-', $user['birthday']);
			if (count($parts) == 3)
			{
				$import->dob_day = trim($parts[0]);
				$import->dob_month = trim($parts[1]);
				$import->dob_year = trim($parts[2]);
			}
		}

		switch ($user['avatartype'])
		{
			case 'upload':
				if (!preg_match('/avatar_\d+\.[a-z0-9_]+/i', $user['avatar'], $match))
				{
					break;
				}
				$basename = $match[0];

				$originalPath = $this->baseConfig['uploads_path'] . "/avatars/$basename";
				if (!file_exists($originalPath))
				{
					break;
				}

				$import->setAvatarPath($originalPath);
				break;
			case 'gravatar':
				if (!preg_match('/gravatar.com\/avatar\/([a-f0-9]{32})\?/i', $user['avatar'], $match))
				{
					break;
				}
				$md5Match = strtolower($match[1]);
				$md5Email = strtolower(md5($user['email']));

				if ($md5Email === $md5Match)
				{
					$import->gravatar = $user['email'];
					break;
				}
				// fall through intentionally, try to fetch as remote
				// no break
			case 'remote':
				$avatarPath = $this->getImageFromUrl($user['avatar']);
				if ($avatarPath)
				{
					$import->setAvatarPath($avatarPath);
				}
				break;
		}

		$import->about = $user['fid2'] ?? '';
		$import->location = $user['fid1'] ?? '';

		$choiceLookUps = $this->session->extra['profileFieldChoices'];
		$fieldValues = [];

		foreach ($this->typeMap('user_field') AS $fieldId => $newFieldId)
		{
			$fieldName = 'fid' . $fieldId;
			$fieldValue = null;

			if (isset($user[$fieldName]) && $user[$fieldName] !== '')
			{
				$sourceFieldValue = $user[$fieldName];

				if (array_key_exists($fieldId, $choiceLookUps))
				{
					$fieldInfo = $choiceLookUps[$fieldId];

					$userFieldValue = [];
					foreach (Arr::stringToArray($user[$fieldName], '/\r?\n/') AS $fieldChoiceId)
					{
						if (isset($fieldInfo[$fieldChoiceId]))
						{
							$userFieldValue[$fieldInfo[$fieldChoiceId]] = $fieldInfo[$fieldChoiceId];
						}
					}
					$fieldValue = array_unique($userFieldValue);
				}
				else
				{
					$fieldValue = $sourceFieldValue;
				}

				$fieldValues[$newFieldId] = $fieldValue;
			}
		}

		$import->setCustomFields($fieldValues);

		if ($import->is_admin && $user['admin_permissions'])
		{
			$isSuperAdmin = in_array($user['uid'], $stepConfig['super_admins']) ? 1 : 0;

			$permissionCache = [];
			$sourcePerms = @unserialize($user['admin_permissions']);
			if ($sourcePerms)
			{
				if (!empty($sourcePerms['config']['settings']))
				{
					$permissionCache[] = 'option';
				}
				if (!empty($sourcePerms['config']['smilies']))
				{
					$permissionCache[] = 'bbCodeSmilie';
				}
				if (!empty($sourcePerms['config']['plugins']))
				{
					$permissionCache[] = 'addOn';
				}
				if (!empty($sourcePerms['forum']['management']))
				{
					$permissionCache[] = 'node';
				}
				if (!empty($sourcePerms['forum']['management']))
				{
					$permissionCache[] = 'thread';
				}
				if (!empty($sourcePerms['forum']['attachments']))
				{
					$permissionCache[] = 'attachment';
				}
				if (!empty($sourcePerms['forum']['announcements']))
				{
					$permissionCache[] = 'notice';
				}
				if (!empty($sourcePerms['user']['users']))
				{
					$permissionCache[] = 'user';
				}
				if (!empty($sourcePerms['user']['users']))
				{
					$permissionCache[] = 'userField';
				}
				if (!empty($sourcePerms['user']['users']))
				{
					$permissionCache[] = 'trophy';
				}
				if (!empty($sourcePerms['user']['users']))
				{
					$permissionCache[] = 'warning';
				}
				if (!empty($sourcePerms['user']['banning']))
				{
					$permissionCache[] = 'ban';
				}
				if (!empty($sourcePerms['user']['group']))
				{
					$permissionCache[] = 'userGroup';
				}
				if (!empty($sourcePerms['user']['users']))
				{
					$permissionCache[] = 'userUpgrade';
				}
				if (!empty($sourcePerms['style']['templates']))
				{
					$permissionCache[] = 'style';
				}
				if (!empty($sourcePerms['style']['templates']))
				{
					$permissionCache[] = 'language';
				}
				if (!empty($sourcePerms['tools']['tasks']))
				{
					$permissionCache[] = 'cron';
				}
				if (!empty($sourcePerms['tools']['cache']))
				{
					$permissionCache[] = 'rebuildCache';
				}
				if (!empty($sourcePerms['tools']['statistics']))
				{
					$permissionCache[] = 'viewStatistics';
				}
				if (!empty($sourcePerms['tools']['modlog']))
				{
					$permissionCache[] = 'viewLogs';
				}
				if (!empty($sourcePerms['forum']['management']))
				{
					$permissionCache[] = 'import';
				}

				$permissionCache[] = 'upgradeXenForo';
			}

			$adminData = [
				'is_super_admin' => $isSuperAdmin,
				'permission_cache' => $permissionCache,
			];

			$import->setAdmin($adminData);
		}

		$optionData = $this->mapXfKeys($user, [
			'content_show_signature' => 'showsigs',
			'receive_admin_email' => 'allownotices',
			'email_on_conversation' => 'pmnotify',
			'push_on_conversation' => 'pmnotify',
		]);

		$import->bulkSetDirect('option', $optionData);

		$import->creation_watch_state = ($user['subscriptionmethod'] == 2 ? 'watch_email' : ($user['subscriptionmethod'] == 1 ? 'watch_no_email' : ''));
		$import->interaction_watch_state = $import->creation_watch_state;

		$import->allow_send_personal_conversation = ($user['receivepms'] ? 'members' : 'none');

		return $import;
	}

	// ############################## STEP: PRIVATE MESSAGES #############################

	public function getStepEndPrivateMessages()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(pmid) FROM privatemessages") ?: 0;
	}

	public function stepPrivateMessages(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$pms = $this->sourceDb->fetchAllKeyed("
			SELECT pms.*, users.username
			FROM privatemessages AS pms
			LEFT JOIN users AS users ON (pms.uid = users.uid)
			WHERE pms.pmid > ? AND pms.pmid <= ?
				AND pms.toid > 0
				AND pms.fromid > 0
			ORDER BY pms.pmid
		", 'pmid', [$state->startAfter, $state->end]);

		if (!$pms)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		foreach ($pms AS $pm)
		{
			$mapUserIds[] = $pm['uid'];
			$mapUserIds[] = $pm['toid'];
			$mapUserIds[] = $pm['fromid'];
		}

		$this->lookup('user', $mapUserIds);

		foreach ($pms AS $oldId => $pm)
		{
			$state->startAfter = $oldId;

			if ($pm['uid'] != $pm['toid'])
			{
				continue;
			}

			$userIdsQuoted = $this->sourceDb->quote([$pm['toid'], $pm['fromid']]);
			$users = $this->sourceDb->fetchPairs("
				SELECT uid, username
				FROM users
				WHERE uid IN ($userIdsQuoted)
			");
			if (count($users) != 2)
			{
				continue;
			}

			$newFromUserId = $this->lookupId('user', $pm['fromid']);
			if (!$newFromUserId)
			{
				continue;
			}

			/** @var ConversationMaster $import */
			$import = $this->newHandler(ConversationMaster::class);

			foreach ($users AS $userId => $username)
			{
				$newUserId = $this->lookupId('user', $userId);
				if (!$newUserId)
				{
					continue;
				}

				if ($pm['uid'] == $userId)
				{
					$lastReadDate = $pm['readtime'];
				}
				else
				{
					$lastReadDate = $pm['dateline'];
				}

				$import->addRecipient($newUserId, 'active', [
					'last_read_date' => $lastReadDate,
				]);
			}

			$fromUserName = $users[$pm['fromid']];

			$import->title = $pm['subject'];
			$import->user_id = $newFromUserId;
			$import->username = $fromUserName;
			$import->start_date = $pm['dateline'];

			/** @var ConversationMessage $messageImport */
			$messageImport = $this->newHandler(ConversationMessage::class);

			$messageImport->message_date = $pm['dateline'];
			$messageImport->user_id = $newFromUserId;
			$messageImport->username = $fromUserName;
			$messageImport->message = $this->convertContentToBbCode($pm['message']);

			$import->addMessage($oldId, $messageImport);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ############################## STEP: FORUMS #############################

	public function stepForums(StepState $state, array $stepConfig)
	{
		$forums = $this->sourceDb->fetchAll("
			SELECT *
			FROM forums
		");
		if (!$forums)
		{
			return $state->complete();
		}

		$nodeTree = [];
		foreach ($forums AS $forum)
		{
			$nodeTree[$forum['pid']][$forum['fid']] = $forum;
		}

		$permissions = $this->sourceDb->fetchAll("
			SELECT *
			FROM forumpermissions
		");

		$permissionsGrouped = [];
		foreach ($permissions AS $permission)
		{
			$permissionsGrouped[$permission['fid']][$permission['gid']] = $permission;
		}

		$state->imported = $this->importNodeTree($forums, $nodeTree, $permissionsGrouped);

		return $state->complete();
	}

	protected function importNodeTree(array $nodes, array $tree, array $permissionsGrouped, $oldParentId = 0)
	{
		if (!isset($tree[$oldParentId]))
		{
			return 0;
		}

		$total = 0;

		foreach ($tree[$oldParentId] AS $node)
		{
			$oldNodeId = $node['fid'];

			/** @var Node $importNode */
			$importNode = $this->newHandler(Node::class);
			$importNode->bulkSet($this->mapXfKeys($node, [
				'title' => 'name',
				'display_order' => 'disporder',
			]));
			$importNode->description = strip_tags($node['description']);
			$importNode->parent_node_id = $this->lookupId('node', $node['pid'], 0);

			if ($node['type'] == 'f')
			{
				$nodeTypeId = 'Forum';

				/** @var Forum $importType */
				$importType = $this->newHandler(Forum::class);
				$importType->bulkSet($this->mapXfKeys($node, [
					'discussion_count' => 'threads',
					'message_count' => 'posts',
					'last_post_date' => 'lastpost',
					'last_post_username' => 'lastposter',
					'last_thread_title' => 'lastpostsubject',
				]));
			}
			else
			{
				$nodeTypeId = 'Category';

				/** @var Category $importType */
				$importType = $this->newHandler(Category::class);
			}

			$importNode->setType($nodeTypeId, $importType);

			$newNodeId = $importNode->save($oldNodeId);
			if ($newNodeId)
			{
				if (!empty($permissionsGrouped[$oldNodeId]))
				{
					$this->setupNodePermissionImport($permissionsGrouped[$oldNodeId], $newNodeId);
				}

				$total++;
				$total += $this->importNodeTree($nodes, $tree, $permissionsGrouped, $oldNodeId);
			}
		}

		return $total;
	}

	protected function setupNodePermissionImport($permissionsByUserGroup, $newNodeId)
	{
		/** @var Permission $permHelper */
		$permHelper = $this->getDataHelper(Permission::class);

		foreach ($permissionsByUserGroup AS $oldGroupId => $perms)
		{
			if ($oldGroupId == 5)
			{
				continue; // skip these. they'll be treated as guests
			}

			$newGroupId = $this->lookupId('user_group', $oldGroupId);
			if (!$newGroupId)
			{
				continue;
			}

			if ($perms['canview'])
			{
				$newPermissions = [
					'general' => ['viewNode' => 'content_allow'],
					'forum' => [],
				];

				$newPermissions['forum']['viewContent'] = ($perms['canviewthreads'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['viewOthers'] = (!$perms['canonlyviewownthreads'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['postThread'] = ($perms['canpostthreads'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['postReply'] = ($perms['canpostreplys'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['editOwnPost'] = ($perms['caneditposts'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['deleteOwnPost'] = ($perms['candeleteposts'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['deleteOwnThread'] = ($perms['candeletethreads'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['viewAttachment'] = ($perms['candlattachments'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['uploadAttachment'] = ($perms['canpostattachments'] ? 'content_allow' : 'reset');
				$newPermissions['forum']['votePoll'] = ($perms['canvotepolls'] ? 'content_allow' : 'reset');
			}
			else
			{
				$newPermissions = [
					'general' => ['viewNode' => 'reset'],
				];

				$permissionsGrouped = $this->getNodePermissionDefinitionsGrouped();

				foreach ($permissionsGrouped AS $permissionGroupId => $permissionDefinitions)
				{
					foreach ($permissionDefinitions AS $permissionId => $permissionDefinition)
					{
						if ($permissionDefinition['permission_type'] == 'flag')
						{
							$newPermissions[$permissionGroupId][$permissionId] = 'reset';
						}
					}
				}
			}

			// now import
			$permHelper->insertContentUserGroupPermissions('node', $newNodeId, $newGroupId, $newPermissions);
		}
	}

	// ############################## STEP: MODERATORS #############################

	public function stepModerators(StepState $state, array $stepConfig)
	{
		$this->typeMap('node');

		$moderators = $this->sourceDb->fetchAll("
			SELECT mods.*
			FROM moderators AS mods
			INNER JOIN users AS users ON (mods.id = users.uid)
			WHERE mods.isgroup = 0
		");
		if (!$moderators)
		{
			return $state->complete();
		}

		$modsGrouped = [];
		foreach ($moderators AS $moderator)
		{
			$modsGrouped[$moderator['id']][$moderator['fid']] = $moderator;
		}

		$this->lookup('user', array_keys($modsGrouped));

		/** @var Moderator $modHelper */
		$modHelper = $this->getDataHelper(Moderator::class);

		foreach ($modsGrouped AS $userId => $forums)
		{
			$newUserId = $this->lookupId('user', $userId);
			if (!$newUserId)
			{
				continue;
			}

			$inserted = false;

			foreach ($forums AS $forumId => $moderator)
			{
				$newNodeId = $this->lookupId('node', $forumId);
				if (!$newNodeId)
				{
					continue;
				}

				$permissions = $this->convertForumPermissionsForUser($moderator);
				$modHelper->importContentModerator($newUserId, 'node', $newNodeId, $permissions);

				$inserted = true;
			}

			if ($inserted)
			{
				$modHelper->importModerator($newUserId, false, [User::GROUP_MOD]);

				$state->imported++;
			}
		}

		return $state->complete();
	}

	protected function convertForumPermissionsForUser(array $moderator)
	{
		$perms = [];

		if ($moderator['caneditposts'])
		{
			$perms['forum']['editAnyPost'] = 'content_allow';
			$perms['forum']['approveUnapprove'] = 'content_allow';
			$perms['forum']['viewModerated'] = 'content_allow';
		}
		if ($moderator['candeleteposts'])
		{
			$perms['forum']['deleteAnyPost'] = 'content_allow';
			$perms['forum']['deleteAnyThread'] = 'content_allow';
			$perms['forum']['viewDeleted'] = 'content_allow';
			$perms['forum']['undelete'] = 'content_allow';
		}
		if ($moderator['canopenclosethreads'])
		{
			$perms['forum']['lockUnlockThread'] = 'content_allow';
			$perms['forum']['stickUnstickThread'] = 'content_allow';
		}
		if ($moderator['canmanagethreads'])
		{
			$perms['forum']['manageAnyThread'] = 'content_allow';
		}

		return $perms;
	}

	// ############################## STEP: THREADS #############################

	public function getStepEndThreads()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(tid)
			FROM threads
		") ?: 0;
	}

	public function stepThreads(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$threads = $this->sourceDb->fetchAllKeyed("
			SELECT threads.*,
				IF(users.username IS NOT NULL, users.username, threads.username) AS username
			FROM threads AS threads FORCE INDEX (PRIMARY)
			LEFT JOIN users AS users ON (threads.uid = users.uid)
			INNER JOIN forums AS forums ON (threads.fid = forums.fid)
			WHERE threads.tid > ? AND threads.tid <= ?
			ORDER BY threads.tid
			LIMIT {$limit}
		", 'tid', [$state->startAfter, $state->end]);

		if (!$threads)
		{
			return $state->complete();
		}

		$this->typeMap('node');
		$this->lookup('user', $this->pluck($threads, 'uid'));

		foreach ($threads AS $oldThreadId => $thread)
		{
			$state->startAfter = $oldThreadId;

			$nodeId = $this->lookupId('node', $thread['fid']);

			if (!$nodeId)
			{
				continue;
			}

			/** @var Thread $import */
			$import = $this->newHandler(Thread::class);

			$import->bulkSet($this->mapXfKeys($thread, [
				'reply_count' => 'replies',
				'view_count' => 'views',
				'sticky',
				'last_post_date' => 'lastpost',
				'post_date' => 'dateline',
			]));

			$import->bulkSet([
				'discussion_open' => ($thread['closed'] ? 0 : 1),
				'node_id' => $nodeId,
				'user_id' => $this->lookupId('user', $thread['uid'], 0),
				'discussion_state' => $thread['visible'] ? 'visible' : 'moderated',
			]);

			$import->set('title', $thread['subject'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$import->set('username', $thread['username'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$import->set('last_post_username', $thread['lastposter'], [EntityEmulator::UNHTML_ENTITIES => true]);

			$subs = $this->sourceDb->fetchPairs("
				SELECT uid, notification
				FROM threadsubscriptions
				WHERE tid = {$oldThreadId}
			");

			if ($subs)
			{
				$this->lookup('user', array_keys($subs));

				foreach ($subs AS $userId => $emailSubscribe)
				{
					$newUserId = $this->lookupId('user', $userId);
					if (!$newUserId)
					{
						continue;
					}

					$import->addThreadWatcher($newUserId, $emailSubscribe);
				}
			}

			if ($newThreadId = $import->save($oldThreadId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ############################## STEP: POSTS #############################

	protected function getThreadIdsForPostsStep($startAfter, $end, $threadLimit)
	{
		return $this->sourceDb->fetchAllColumn("
			SELECT tid
			FROM threads
			WHERE tid > ? AND tid <= ?
			ORDER BY tid
			LIMIT {$threadLimit}
		", [$startAfter, $end]);
	}

	protected function getPostsForPostsStep($threadId, $startDate)
	{
		$limit = self::$postsStepLimit;

		// note: visible -2 is a draft so we are skipping these entirely

		return $this->sourceDb->fetchAll("
			SELECT posts.*,
				IF(users.username IS NOT NULL, users.username, posts.username) AS username
			FROM posts AS posts
			LEFT JOIN users AS users ON (posts.uid = users.uid)
			WHERE posts.tid = ?
				AND posts.dateline > ?
				AND posts.visible <> -2
			ORDER BY posts.dateline
			LIMIT {$limit}
		", [$threadId, $startDate]);
	}

	protected function lookupUsers(array $posts)
	{
		$this->lookup('user', $this->pluck($posts, 'uid'));
	}

	protected function getPostDateField()
	{
		return 'dateline';
	}

	protected function getPostIdField()
	{
		return 'pid';
	}

	protected function handlePostImport(array $post, $newThreadId, StepState $state)
	{
		/** @var Post $import */
		$import = $this->newHandler(Post::class);

		$import->bulkSet([
			'thread_id' => $newThreadId,
			'user_id' => $this->lookupId('user', $post['uid']),
			'post_date' => $post['dateline'],
			'message' => $this->convertContentToBbCode($post['message']),
			'message_state' => $post['visible'] ? 'visible' : 'moderated',
			'position' => $state->extra['postPosition'],
		]);

		$import->set('username', $post['username'], [EntityEmulator::UNHTML_ENTITIES => true]);

		$import->setLoggedIp($post['ipaddress']);

		return $import;
	}

	// ########################### STEP: POLLS ###############################

	public function getStepEndPolls()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(pid)
			FROM polls
		") ?: 0;
	}

	public function stepPolls(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$polls = $this->sourceDb->fetchAllKeyed("
			SELECT polls.*
			FROM polls AS polls
			INNER JOIN threads AS thread ON (thread.tid = polls.tid)
			WHERE polls.pid > ? AND polls.pid <= ?
			ORDER BY polls.pid
			LIMIT {$limit}
		", 'pid', [$state->startAfter, $state->end]);

		if (!$polls)
		{
			return $state->complete();
		}

		$pollsCompleted = [];

		$this->lookup('thread', $this->pluck($polls, 'tid'));

		foreach ($polls AS $oldId => $poll)
		{
			$state->startAfter = $oldId;

			$newThreadId = $this->lookupId('thread', $poll['tid']);
			if (!$newThreadId)
			{
				continue;
			}

			if (array_key_exists($oldId, $pollsCompleted))
			{
				// poll id in the thread table isn't unique, so use this to avoid duplication
				continue;
			}

			$pollsCompleted[$oldId] = true;

			/** @var Poll $import */
			$import = $this->newHandler(Poll::class);
			$import->set('question', $poll['question'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$import->bulkSet([
				'content_type' => 'thread',
				'content_id' => $newThreadId,
				'public_votes' => $poll['public'],
				'max_votes' => $poll['multiple'] ? 0 : 1,
				'close_date' => ($poll['timeout'] ? $poll['dateline'] + 86400 * $poll['timeout'] : 0),
			]);

			$responses = explode('||~|~||', $this->convertToUtf8($poll['options'], null, true));
			if (end($responses) === '')
			{
				array_pop($responses);
			}
			if (!$responses)
			{
				continue;
			}

			$importResponses = [];

			foreach ($responses AS $i => $responseText)
			{
				/** @var PollResponse $importResponse */
				$importResponse = $this->newHandler(PollResponse::class);
				$importResponse->preventRetainIds();
				$importResponse->response = $responseText;

				$importResponses[$i] = $importResponse;

				$import->addResponse($i, $importResponse);
			}

			$votes = $this->sourceDb->fetchAll("
				SELECT uid, dateline, voteoption
				FROM pollvotes
				WHERE pid = ?
			", $poll['pid']);

			$this->lookup('user', $this->pluck($votes, 'uid'));

			foreach ($votes AS $vote)
			{
				$voteUserId = $this->lookupId('user', $vote['uid']);
				if (!$voteUserId)
				{
					continue;
				}

				$voteOption = max(0, $vote['voteoption'] - 1);

				if (!array_key_exists($voteOption, $importResponses))
				{
					continue;
				}

				$importResponses[$voteOption]->addVote($voteUserId, $vote['dateline']);
			}

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: ATTACHMENTS ###############################

	public function getStepEndAttachments()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(aid)
			FROM attachments
		") ?: 0;
	}

	public function stepAttachments(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$attachments = $this->sourceDb->fetchAll("
			SELECT *
			FROM attachments
			WHERE aid > ? AND aid <= ?
				AND visible = 1
				AND pid > 0
			ORDER BY aid
			LIMIT {$limit}
		", [$state->startAfter, $state->end]);

		if (!$attachments)
		{
			return $state->complete();
		}

		$this->lookup('user', $this->pluck($attachments, 'uid'));
		$this->lookup('post', $this->pluck($attachments, 'pid'));

		$uploadsPath = $this->baseConfig['uploads_path'];

		foreach ($attachments AS $attachment)
		{
			$state->startAfter = $attachment['aid'];

			$newPostId = $this->lookupId('post', $attachment['pid']);
			if (!$newPostId)
			{
				continue;
			}

			$attachPath = $uploadsPath . \XF::$DS . $attachment['attachname'];
			if (!file_exists($attachPath))
			{
				continue;
			}

			/** @var Attachment $import */
			$import = $this->newHandler(Attachment::class);
			$import->bulkSet([
				'content_type' => 'post',
				'content_id'   => $newPostId,
				'attach_date'  => $attachment['dateuploaded'],
				'view_count'   => $attachment['downloads'],
				'unassociated' => false,
			]);
			$import->setDataExtra('upload_date', $attachment['dateuploaded']);

			$import->setDataUserId($this->lookupId('user', $attachment['uid']));
			$import->setSourceFile($attachPath, $this->convertToUtf8($attachment['filename'], null, true));
			$import->setContainerCallback([$this, 'rewriteEmbeddedAttachments']);

			$newId = $import->save($attachment['aid']);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		File::cleanUpTempFiles();

		return $state->resumeIfNeeded();
	}

	protected function convertContentToBbCode($content, $autoLink = true)
	{
		if ($autoLink)
		{
			$bbCodeContainer = $this->app->bbCode();
			$parser = $bbCodeContainer->parser();
			$rules = $bbCodeContainer->rules('import:mybb');

			$processor = $bbCodeContainer->processor();

			/** @var AutoLink $autoLinker */
			$autoLinker = $bbCodeContainer->processorAction('autolink');
			$autoLinker->enableUnfurling(false);

			$processor->addProcessorAction('autolink', $autoLinker);

			$content = $processor->render($content, $parser, $rules);
		}

		$content = preg_replace('#\[align=left](.*)\[/align]#siU', '[LEFT]$1[/LEFT]', $content);
		$content = preg_replace('#\[align=center](.*)\[/align]#siU', '[CENTER]$1[/CENTER]', $content);
		$content = preg_replace('#\[align=right](.*)\[/align]#siU', '[RIGHT]$1[/RIGHT]', $content);
		$content = preg_replace('#\[align=justify](.*)\[/align]#siU', '[LEFT]$1[/LEFT]', $content);

		$content = preg_replace('#(\[quote=[\'|"][^\'|"]*[\'|"])[^]]*]#iU', '$1]', $content);

		$content = preg_replace('#\[size=xx-small](.*)\[/size]#siU', '[SIZE=1]$1[/SIZE]', $content);
		$content = preg_replace('#\[size=x-small](.*)\[/size]#siU', '[SIZE=2]$1[/SIZE]', $content);
		$content = preg_replace('#\[size=small](.*)\[/size]#siU', '[SIZE=3]$1[/SIZE]', $content);
		$content = preg_replace('#\[size=medium](.*)\[/size]#siU', '[SIZE=4]$1[/SIZE]', $content);
		$content = preg_replace('#\[size=large](.*)\[/size]#siU', '[SIZE=5]$1[/SIZE]', $content);
		$content = preg_replace('#\[size=x-large](.*)\[/size]#siU', '[SIZE=6]$1[/SIZE]', $content);
		$content = preg_replace('#\[size=xx-large](.*)\[/size]#siU', '[SIZE=7]$1[/SIZE]', $content);

		$content = preg_replace('#\[attachment=(\d)+]#siU', '[ATTACH=full]$1[/ATTACH]', $content);

		return $content;
	}
}
