<?php

namespace XFI\Import\Importer;

use XF\Db\Exception;
use XF\Db\Mysqli\Adapter;
use XF\Entity\User;
use XF\Import\Data\Attachment;
use XF\Import\Data\Category;
use XF\Import\Data\ConversationMaster;
use XF\Import\Data\ConversationMessage;
use XF\Import\Data\EntityEmulator;
use XF\Import\Data\Forum;
use XF\Import\Data\LinkForum;
use XF\Import\Data\Node;
use XF\Import\Data\Poll;
use XF\Import\Data\PollResponse;
use XF\Import\Data\Post;
use XF\Import\Data\Thread;
use XF\Import\Data\UserField;
use XF\Import\Data\UserGroup;
use XF\Import\DataHelper\Moderator;
use XF\Import\DataHelper\Permission;
use XF\Import\Importer\AbstractForumImporter;
use XF\Import\Importer\StepPostsTrait;
use XF\Import\StepState;
use XF\Timer;
use XF\Util\Arr;
use XF\Util\File;
use XF\Validator\Url;

use function array_key_exists, count, in_array, intval, is_array, strval;

class SMF extends AbstractForumImporter
{
	use StepPostsTrait;

	/**
	 * @var Adapter
	 */
	protected $sourceDb;

	public static function getListInfo()
	{
		return [
			'target' => 'XenForo',
			'source' => 'SMF 2.1',
		];
	}

	protected function getBaseConfigDefault()
	{
		return [
			'db' => [
				'host' => '',
				'username' => '',
				'password' => '',
				'dbname' => '',
				'port' => 3306,
				'tablePrefix'   => '',
				'charset'  => '', // used for the DB connection
			],
			'charset' => '',
		];
	}

	public function renderBaseConfigOptions(array $vars)
	{
		$vars['db'] = [
			'host' => $this->app->config['db']['host'],
			'port' => $this->app->config['db']['port'],
			'username' => $this->app->config['db']['username'],
			'tablePrefix' => 'smf_',
		];
		return $this->app->templater()->renderTemplate('admin:xfi_import_config_smf', $vars);
	}

	public function validateBaseConfig(array &$baseConfig, array &$errors)
	{
		$baseConfig['db']['tablePrefix'] = preg_replace('/[^a-z0-9_]/i', '', $baseConfig['db']['tablePrefix']);

		$fullConfig = array_replace_recursive($this->getBaseConfigDefault(), $baseConfig);
		$missingFields = false;

		if ($fullConfig['db']['host'])
		{
			$validDbConnection = false;

			try
			{
				$sourceDb = new Adapter($fullConfig['db'], false);
				$sourceDb->getConnection();
				$validDbConnection = true;
			}
			catch (Exception $e)
			{
				$errors[] = \XF::phrase('source_database_connection_details_not_correct_x', ['message' => $e->getMessage()]);
			}

			if ($validDbConnection)
			{
				try
				{
					$sourceDb->fetchOne("
						SELECT id_member
						FROM members
						LIMIT 1
					");
				}
				catch (Exception $e)
				{
					if ($fullConfig['db']['dbname'] === '')
					{
						$errors[] = \XF::phrase('please_enter_database_name');
					}
					else
					{
						$errors[] = \XF::phrase('table_prefix_or_database_name_is_not_correct');
					}
				}
			}
			else
			{
				$missingFields = true;
			}
		}

		if ($missingFields)
		{
			$errors[] = \XF::phrase('please_complete_required_fields');
		}

		return $errors ? false : true;
	}

	protected function getStepConfigDefault()
	{
		return [
			'users' => [
				'merge_email' => true,
				'merge_name'  => false,
				'server_avatar_path' => '',
				'avatar_path' => '',
			],
			'attachments' => [
				'attach_paths' => [],
			],
		];
	}

	public function renderStepConfigOptions(array $vars)
	{
		$vars['stepConfig'] = $this->getStepConfigDefault();

		$sourceDb = new Adapter(
			$this->baseConfig['db']
		);

		if (in_array('users', $vars['steps']))
		{
			$serverStoredAvatarDir = $sourceDb->fetchOne("
				SELECT value
				FROM settings
				WHERE variable = 'avatar_directory'
			");
			$vars['stepConfig']['users']['server_avatar_path'] = $serverStoredAvatarDir;

			$customAvatarDirEnabled = $sourceDb->fetchOne("
				SELECT value
				FROM settings
				WHERE variable = 'custom_avatar_enabled'
			");
			if ($customAvatarDirEnabled)
			{
				$customAvatarDir = $sourceDb->fetchOne("
					SELECT value
					FROM settings
					WHERE variable = 'custom_avatar_dir'
				");
				$vars['stepConfig']['users']['avatar_path'] = $customAvatarDir;
			}
		}

		if (in_array('attachments', $vars['steps']))
		{
			$attachPaths = [];

			$uploadDir = $sourceDb->fetchOne("
				SELECT value
				FROM settings
				WHERE variable = 'attachmentUploadDir'
			");

			$uploadDirs = @json_decode($uploadDir, true);
			if (is_array($uploadDirs)) // multiple upload directories configured
			{
				$attachPaths = $uploadDirs;
			}
			else // single path
			{
				$currentDirId = $sourceDb->fetchOne("
					SELECT value
					FROM settings
					WHERE variable = 'currentAttachmentUploadDir'
				") ?: 1;
				$attachPaths[$currentDirId] = $uploadDir;
			}

			$vars['stepConfig']['attachments']['attach_paths'] = $attachPaths;
		}

		return $this->app->templater()->renderTemplate('admin:xfi_import_step_config_smf', $vars);
	}

	public function validateStepConfig(array $steps, array &$stepConfig, array &$errors)
	{
		foreach (array_keys($stepConfig) AS $step)
		{
			$callback = 'validateStepConfig' . ucfirst($step);

			if (method_exists($this, $callback))
			{
				$this->$callback($stepConfig[$step], $stepConfig, $errors);
			}
		}

		return $errors ? false : true;
	}

	protected function validateStepConfigUsers(array &$config, array &$stepConfig, array &$errors)
	{
		if (!isset($stepConfig['users']['avatar_path']))
		{
			return true; // skip avatars then
		}

		$path = $stepConfig['users']['avatar_path'];

		if (!file_exists($path) || !is_readable($path))
		{
			$errors[] = \XF::phrase('directory_x_not_found_is_not_readable', ['dir' => $path]);
			return false;
		}

		return true;
	}

	protected function validateStepConfigAttachments(array &$config, array &$stepConfig, array &$errors)
	{
		if (!isset($stepConfig['attachments']['attach_paths']))
		{
			return true; // skip attachments then
		}

		foreach ($stepConfig['attachments']['attach_paths'] AS $folderId => $path)
		{
			if (!$path)
			{
				unset($stepConfig['attachments']['attach_paths'][$folderId]);
				continue;
			}

			if (!file_exists($path) || !is_readable($path))
			{
				$errors[] = \XF::phrase('directory_x_not_found_is_not_readable', ['dir' => $path]);
				return false;
			}
		}

		return true;
	}

	public function getSteps()
	{
		return [
			'userGroups' => [
				'title' => \XF::phrase('user_groups'),
			],
			'userFields' => [
				'title' => \XF::phrase('custom_user_fields'),
			],
			'users' => [
				'title' => \XF::phrase('users'),
				'depends' => ['userGroups', 'userFields'],
				'force' => ['attachments'], // yep - avatars are stored as attachments
			],
			'privateMessages' => [
				'title' => \XF::phrase('xfi_import_private_messages'),
				'depends' => ['users'],
			],
			'forums' => [
				'title' => \XF::phrase('forums'),
			],
			'moderators' => [
				'title' => \XF::phrase('moderators'),
				'depends' => ['forums', 'users'],
			],
			'threads' => [
				'title' => \XF::phrase('threads'),
				'depends' => ['forums', 'users'],
				'force' => ['posts'],
			],
			'posts' => [
				'title' => \XF::phrase('posts'),
				'depends' => ['threads'],
			],
			'attachments' => [
				'title' => \XF::phrase('attachments'),
				'depends' => ['posts'], // can come from conversations as well, though not required
			],
		];
	}

	protected function doInitializeSource()
	{
		$dbConfig = $this->baseConfig['db'];

		$this->sourceDb = new Adapter($dbConfig, false);

		$this->dataManager->setSourceCharset($this->baseConfig['charset'], true);
	}

	// ############################## STEP: USER GROUPS #########################

	public function stepUserGroups(StepState $state, array $stepConfig)
	{
		$groups = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM membergroups
			ORDER BY id_group
		", 'id_group');

		foreach ($groups AS $oldId => $group)
		{
			$permissions = $this->calculateGroupPerms($oldId);
			$titlePriority = 5;

			$groupMap = [
				1 => User::GROUP_ADMIN,

				2 => User::GROUP_MOD,
				3 => User::GROUP_MOD,

				4 => User::GROUP_REG,
			];

			if (array_key_exists($oldId, $groupMap))
			{
				// don't import the group, just map it to one of our defaults
				$this->logHandler(UserGroup::class, $oldId, $groupMap[$oldId]);
			}
			else
			{
				if ($oldId == 2) // super mods
				{
					$titlePriority = 910;
				}

				/** @var UserGroup $import */
				$import = $this->newHandler(UserGroup::class);
				$import->title = $group['group_name'];
				$import->display_style_priority = $titlePriority;
				$import->setPermissions($permissions);
				$import->save($oldId);
			}

			$state->imported++;
		}

		return $state->complete();
	}

	protected function calculateGroupPerms($groupId)
	{
		$oldPerms = $this->getGroupPermissions($groupId);
		$perms = [];

		// no equivalents
		$perms['general']['view'] = 'allow';
		$perms['general']['viewNode'] = 'allow';
		$perms['general']['maxMentionedUsers'] = 5;
		$perms['general']['report'] = 'allow';
		$perms['general']['editSignature'] = 'allow';
		$perms['forum']['viewOthers'] = 'allow';
		$perms['forum']['viewContent'] = 'allow';
		$perms['forum']['react'] = 'allow';
		$perms['forum']['votePoll'] = 'allow';

		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'viewMemberList', 'view_mlist');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'viewProfile', 'profile_view_any');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'search', 'search_posts');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'editProfile', 'profile_extra_own');
		$this->convertOldPermissionValue($perms, $oldPerms, 'general', 'editCustomTitle', 'profile_title_own');
		$this->convertOldPermissionValue($perms, $oldPerms, 'avatar', 'allow', 'profile_upload_avatar');
		$this->convertOldPermissionValue($perms, $oldPerms, 'conversation', 'receive', 'pm_read');
		$this->convertOldPermissionValue($perms, $oldPerms, 'conversation', 'start', 'pm_send');

		if (!empty($perms['general']['viewProfile'])
			&& $perms['general']['viewProfile'] == 'allow'
		)
		{
			$perms['profilePost']['view'] = 'allow';
			$perms['profilePost']['react'] = 'allow';
			$perms['profilePost']['manageOwn'] = 'allow';
			$perms['profilePost']['deleteOwn'] = 'allow';
			$perms['profilePost']['post'] = 'allow';
			$perms['profilePost']['comment'] = 'allow';
			$perms['profilePost']['editOwn'] = 'allow';
		}

		return $perms;
	}

	protected function convertOldPermissionValue(array &$outputPerms, array $oldPerms, $newGroup, $newId, $oldId, $allow = 'allow')
	{
		if (!isset($oldPerms[$oldId]) || $oldPerms[$oldId] == -1)
		{
			return;
		}

		if ($oldPerms[$oldId] == 0)
		{
			$outputPerms[$newGroup][$newId] = 'deny';
		}
		else if ($oldPerms[$oldId] == 1)
		{
			$outputPerms[$newGroup][$newId] = $allow;
		}
	}

	protected function getGroupPermissions($groupId)
	{
		return $this->sourceDb->fetchPairs('
			SELECT permission, add_deny
			FROM permissions
			WHERE id_group = ?
		', $groupId);
	}

	// ############################## STEP: USER FIELDS #########################

	public function stepUserFields(StepState $state, array $stepConfig)
	{
		$fields = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM custom_fields
			ORDER BY id_field
		", 'id_field');

		$choiceLookUps = [];
		$existingFields = $this->db()->fetchPairs("SELECT field_id, field_id FROM xf_user_field");

		$displayOrder = 1;
		foreach ($fields AS $oldId => $field)
		{
			$fieldId = $this->convertToUniqueId($field['col_name'], $existingFields, 25);
			$displayOrder *= 100;

			$data = [
				'field_id' => $fieldId,
				'display_order' => $displayOrder,
				'max_length' => $field['field_length'],
				'viewable_profile' => $field['show_display'],
				'show_registration' => $field['show_reg'] ? true : false,
				'required' => $field['show_reg'] == 2 ? true : false,
			];

			switch ($field['field_type'])
			{
				case 'select':
				case 'radio':
					$data['field_type'] = $field['field_type'];

					$values = Arr::stringToArray($field['field_options'], '/,/');
					$data['field_choices'] = $values;
					$choiceLookUps[$oldId] = $values;
					break;

				case 'check':
					$data['field_type'] = 'checkbox';

					$values = [1 => \XF::phrase('yes')->render()];
					$data['field_choices'] = $values;
					$choiceLookUps[$oldId] = $values;
					break;

				case 'text':
				case 'textarea':
					if ($field['bbc'])
					{
						$data['field_type'] = 'bbcode';
					}
					else
					{
						$data['field_type'] = $field['field_type'] == 'textarea' ? 'textarea' : 'textbox';
					}
					break;
			}

			/** @var UserField $import */
			$import = $this->newHandler(UserField::class);
			$import->setTitle($field['field_name'], $field['field_desc']);
			$import->bulkSet($data);

			$import->save($oldId);

			$state->imported++;
		}

		$this->session->extra['profileFieldChoices'] = $choiceLookUps;

		return $state->complete();
	}

	// ############################## STEP: USERS #############################

	public function getStepEndUsers()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(id_member) FROM members") ?: 0;
	}

	public function stepUsers(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$users = $this->sourceDb->fetchAllKeyed("
			SELECT members.*,
				themes_location.value AS location,
				user_alerts_prefs_announcements.alert_value AS notify_announcements,
				user_alerts_prefs_pm_new.alert_value AS pm_email_notify
			FROM members AS members
			LEFT JOIN themes AS themes_location
				ON (
					themes_location.id_theme = 1 AND
					themes_location.id_member = members.id_member AND
					themes_location.variable = 'cust_loca'
				)
			LEFT JOIN user_alerts_prefs AS user_alerts_prefs_announcements
				ON (
					user_alerts_prefs_announcements.id_member = members.id_member AND
					user_alerts_prefs_announcements.alert_pref = 'announcements'
				)
			LEFT JOIN user_alerts_prefs AS user_alerts_prefs_pm_new
				ON (
					user_alerts_prefs_pm_new.id_member = members.id_member AND
					user_alerts_prefs_pm_new.alert_pref = 'pm_new'
				)
			WHERE members.id_member > ? AND members.id_member <= ?
			ORDER BY members.id_member
			LIMIT {$limit}
		", 'id_member', [$state->startAfter, $state->end]);

		if (!$users)
		{
			return $state->complete();
		}

		$this->typeMap('user_group');

		foreach ($users AS $oldId => $user)
		{
			$state->startAfter = $oldId;

			$import = $this->setupImportUser($user, $state, $stepConfig);
			if ($this->importUser($oldId, $import, $stepConfig))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	protected function setupImportUser(array $user, StepState $state, array $stepConfig)
	{
		/** @var \XF\Import\Data\User $import */
		$import = $this->newHandler(\XF\Import\Data\User::class);

		$userData = $this->mapXfKeys($user, [
			'username' => 'member_name',
			'email' => 'email_address',
			'last_activity' => 'last_login',
			'register_date' => 'date_registered',
			'message_count' => 'posts',
			'visible' => 'show_online',
		]);

		$import->bulkSetDirect('user', $userData);

		$import->user_state = $user['is_activated'] == 1 ? 'valid' : 'moderated';

		$import->user_group_id = User::GROUP_REG;

		$groupIds = [];
		if (!empty($user['additional_groups']))
		{
			$groupIds = explode(',', $user['additional_groups']);
		}
		$groupIds[] = $user['id_group'];
		$groupIds[] = $user['id_post_group'];

		$import->secondary_group_ids = $this->lookup('user_group', $groupIds);

		$import->is_admin = $user['id_group'] == 1 ? 1 : 0;
		if ($import->is_admin)
		{
			// no real concept of admin permissions so import as super admins
			$adminData = [
				'is_super_admin' => true,
				'permission_cache' => [],
			];

			$import->setAdmin($adminData);
		}

		$userBan = $this->getUserBan($user);
		$import->is_banned = $userBan ? 1 : 0;
		if ($import->is_banned)
		{
			$import->setBan([
				'ban_user_id' => 0,
				'ban_date' => $userBan['ban_time'],
				'end_date' => $userBan['expire_time'],
				'user_reason' => $userBan['reason'],
			]);
		}

		// This is pretty much a best effort though should be roughly accurate based on what we know.
		$timezone = $this->sourceDb->fetchOne('
			SELECT value
			FROM settings
			WHERE variable = \'default_timezone\'
		');
		if ($timezone)
		{
			$timezone = explode('GMT', $timezone);
			if (isset($timezone[1]))
			{
				$timezone = $timezone[1];
			}
			else
			{
				$timezone = 0;
			}
		}
		else
		{
			$timezone = 0;
		}

		$timezoneOffset = $timezone + $user['time_offset'];
		$import->timezone = $this->getTimezoneFromOffset($timezoneOffset, false);

		$import->setPasswordData('XF:SMF', [
			'hash' => $user['passwd'],
			'username' => strtolower($user['member_name']),
		]);

		$import->setRegistrationIp($user['member_ip']);

		$import->website = $user['website_url'];
		$import->signature = $this->convertContentToBbCode($user['signature']);

		if ($user['birthdate'])
		{
			$parts = explode('-', $user['birthdate']);
			if (count($parts) == 3)
			{
				// Default birth year.
				if (trim($parts[0]) != '0001')
				{
					$import->dob_day = trim($parts[2]);
					$import->dob_month = trim($parts[1]);

					if (trim($parts[0] != '0004'))
					{
						$import->dob_year = trim($parts[0]);
					}
				}
			}
		}

		$originalPath = null;

		if ($user['avatar'])
		{
			/** @var Url $validator */
			$validator = $this->app->validator('Url');
			$avatar = $validator->coerceValue($user['avatar']);
			if ($validator->isValid($avatar))
			{
				$originalPath = $this->getImageFromUrl($avatar);
			}
			else if ($stepConfig['server_avatar_path'])
			{
				// we'll try and see if it matches an avatar gallery entry
				$originalPath = "{$stepConfig['server_avatar_path']}/{$user['avatar']}";
			}
		}
		else
		{
			$attachment = $this->sourceDb->fetchRow("
				SELECT *
				FROM attachments
				WHERE id_member = ?
				AND id_msg = 0
			", [$user['id_member']]);

			if ($attachment)
			{
				$originalPath = $this->getAttachmentFilename($attachment);
			}
		}

		if ($originalPath && file_exists($originalPath))
		{
			$import->setAvatarPath($originalPath);
		}

		$import->about = $user['personal_text'];
		$import->location = $user['location'] ?? '';

		$choiceLookUps = $this->session->extra['profileFieldChoices'];
		$fieldValues = [];

		$userFieldMap = $this->typeMap('user_field');
		if ($userFieldMap)
		{
			$fields = $this->sourceDb->fetchAllKeyed("
				SELECT field.*, theme.value
				FROM custom_fields AS field
				INNER JOIN themes AS theme ON
					(field.col_name = theme.variable AND theme.id_member = ?)
				WHERE field.id_field IN(" . $this->sourceDb->quote(array_keys($userFieldMap)) . ")
				ORDER BY field.id_field
			", 'id_field', [$user['id_member']]);

			foreach ($fields AS $oldId => $field)
			{
				$newId = $this->lookupId('user_field', $oldId);
				if (!$newId)
				{
					continue;
				}

				$fieldValue = '';

				if ($field['value'])
				{
					if (isset($choiceLookUps[$oldId]))
					{
						if ($field['field_type'] == 'check')
						{
							$fieldValue = [$field['value'] => $field['value']];
						}
						else
						{
							$fieldInfo = $choiceLookUps[$oldId];
							$fieldValue = array_search($field['value'], $fieldInfo);
						}
					}
					else
					{
						if ($field['bbc'])
						{
							$fieldValue = $this->convertContentToBbCode($field['value']);
						}
						else
						{
							$fieldValue = $field['value'];
						}
					}
				}

				$fieldValues[$newId] = $fieldValue;
			}
		}

		$import->setCustomFields($fieldValues);

		$import->receive_admin_email = !empty($user['notify_announcements']) ? 1 : 0;
		$import->email_on_conversation = !empty($user['pm_email_notify']) ? 1 : 0;
		$import->push_on_conversation = !empty($user['pm_email_notify']) ? 1 : 0;
		$import->content_show_signature = true;
		$import->allow_send_personal_conversation = 'members';

		return $import;
	}

	protected function getUserBan(array $user)
	{
		return $this->sourceDb->fetchRow("
			SELECT banitems.*, bangroups.*
			FROM ban_items AS banitems
			INNER JOIN ban_groups AS bangroups ON
				(banitems.id_ban_group = bangroups.id_ban_group)
			WHERE banitems.id_member = ?
				OR (banitems.email_address = ? AND banitems.email_address <> '')
			ORDER BY bangroups.ban_time DESC
		", [$user['id_member'], $user['email_address']]);
	}

	// ############################## STEP: PRIVATE MESSAGES #############################

	public function getStepEndPrivateMessages()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(id_pm) FROM personal_messages") ?: 0;
	}

	public function stepPrivateMessages(StepState $state, array $stepConfig, $maxTime, $limit = 500)
	{
		$timer = new Timer($maxTime);

		$pms = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM personal_messages
			WHERE id_pm > ? AND id_pm <= ?
			ORDER BY id_pm
			LIMIT {$limit}
		", 'id_pm', [$state->startAfter, $state->end]);

		if (!$pms)
		{
			return $state->complete();
		}

		foreach ($pms AS $oldId => $pm)
		{
			$state->startAfter = $oldId;

			$toUsers = $this->sourceDb->fetchPairs('
				SELECT recip.id_member, member.member_name
				FROM pm_recipients AS recip
				INNER JOIN members AS member ON
					(recip.id_member = member.id_member)
				WHERE recip.id_pm = ?
					AND recip.bcc = 0
			', $oldId);
			// we have no concept of BCC, so best to not import these recipients as it leaks info

			if (!$toUsers && !$pm['id_member_from'])
			{
				continue;
			}

			$users = [
				$pm['id_member_from'] => $pm['from_name'],
			] + $toUsers;

			$this->lookup('user', array_keys($users));

			$newFromUserId = $this->lookupId('user', $pm['id_member_from'], 0);

			$recipients = $this->sourceDb->fetchAllKeyed('
				SELECT *
				FROM pm_recipients
				WHERE id_pm = ?
					AND bcc = 0
			', 'id_member', $pm['id_pm']);

			/** @var ConversationMaster $import */
			$import = $this->newHandler(ConversationMaster::class);

			foreach ($users AS $userId => $username)
			{
				$newUserId = $this->lookupId('user', $userId);
				if (!$newUserId)
				{
					continue;
				}

				if (isset($recipients[$userId]))
				{
					$isRead = $recipients[$userId]['is_read'];

					$lastReadDate = ($isRead ? $pm['msgtime'] : 0);
					$isUnread = $isRead ? 0 : 1;
					$deleted = !empty($recipients[$userId]['deleted']);
				}
				else
				{
					$lastReadDate = $pm['msgtime'];
					$isUnread = 0;

					if ($userId == $pm['id_member_from'])
					{
						$deleted = $pm['deleted_by_sender'] ? true : false;
					}
					else
					{
						$deleted = true;
					}
				}

				$import->addRecipient($newUserId, $deleted ? 'deleted' : 'active', [
					'last_read_date' => $lastReadDate,
					'is_unread' => $isUnread,
				]);
			}

			$fromUserName = $pm['from_name'];

			$import->title = $pm['subject'];
			$import->user_id = $newFromUserId;
			$import->username = $fromUserName;
			$import->start_date = $pm['msgtime'];

			/** @var ConversationMessage $messageImport */
			$messageImport = $this->newHandler(ConversationMessage::class);

			$messageImport->message_date = $pm['msgtime'];
			$messageImport->user_id = $newFromUserId;
			$messageImport->username = $fromUserName;
			$messageImport->message = $this->convertContentToBbCode($pm['body']);

			$import->addMessage($oldId, $messageImport);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ############################## STEP: FORUMS #############################

	public function stepForums(StepState $state, array $stepConfig)
	{
		$forums = $this->sourceDb->fetchAll('
			SELECT id_board, id_cat, id_parent,
				board_order, name, member_groups,
				description, redirect, num_posts,
				num_topics, id_profile
			FROM boards
			ORDER BY id_board
		');
		$categories = $this->sourceDb->fetchAll('
			SELECT *
			FROM categories
			ORDER BY id_cat
		');

		$nodes = [];

		foreach ($forums AS $forum)
		{
			$nodes[$forum['id_board']] = $forum['id_board'];
		}

		// category / board IDs can overlap so track their changes here
		$categoryIdMap = [0 => 0];

		foreach ($categories AS $category)
		{
			$newCatId = $category['id_cat'];
			while (isset($nodes[$newCatId]))
			{
				$newCatId++;
			}
			$categoryIdMap[$category['id_cat']] = $newCatId;
			$nodes[$newCatId] = $newCatId;

			$forums[] = [
				'id_board' => $newCatId,
				'id_cat' => 0,
				'id_parent' => 0,
				'board_order' => $category['cat_order'],
				'name' => $category['name'],
				'description' => '',
				'redirect' => '',
				'num_posts' => 0,
				'num_topics' => 0,
				'id_profile' => 0,
				'member_groups' => '',
				'old_cat_id' => $category['id_cat'],
			];
		}

		if (!$forums)
		{
			return $state->complete();
		}

		$nodeTree = [];
		$permissionsGrouped = [];

		foreach ($forums AS $forum)
		{
			$parentId = $forum['id_parent'] ?: $categoryIdMap[$forum['id_cat']];

			$nodeTree[$parentId][$forum['id_board']] = $forum;

			$permSql = $this->sourceDb->query('
				SELECT *
				FROM board_permissions
				WHERE id_profile = ?
			', $forum['id_profile']);
			while ($permission = $permSql->fetch())
			{
				$permissionsGrouped[$forum['id_board']][$permission['id_group']][$permission['permission']] = $permission;
			}
		}

		$state->imported = $this->importNodeTree($forums, $nodeTree, $categoryIdMap, $permissionsGrouped);

		return $state->complete();
	}

	protected function importNodeTree(array $nodes, array $tree, array $categoryIdMap, array $permissionsGrouped, $oldParentId = 0)
	{
		if (!isset($tree[$oldParentId]))
		{
			return 0;
		}

		$total = 0;

		foreach ($tree[$oldParentId] AS $node)
		{
			$oldNodeId = $node['id_board'];

			/** @var Node $importNode */
			$importNode = $this->newHandler(Node::class);
			$importNode->bulkSet($this->mapXfKeys($node, [
				'title' => 'name',
				'display_order' => 'board_order',
			]));
			$importNode->description = $this->convertContentToBbCode($node['description']);
			$importNode->parent_node_id = $this->lookupId('node', $node['id_parent'] ?: $oldParentId, 0);
			$importNode->display_in_list = 1;

			if ($node['redirect'])
			{
				$nodeTypeId = 'LinkForum';

				/** @var LinkForum $importType */
				$importType = $this->newHandler(LinkForum::class);
				$importType->link_url = $node['redirect'];
			}
			else if (!empty($node['old_cat_id']) && isset($categoryIdMap[$node['old_cat_id']]))
			{
				$nodeTypeId = 'Category';

				/** @var Category $importType */
				$importType = $this->newHandler(Category::class);
			}
			else
			{
				$nodeTypeId = 'Forum';

				/** @var Forum $importType */
				$importType = $this->newHandler(Forum::class);
				$importType->bulkSet($this->mapXfKeys($node, [
					'discussion_count' => 'num_topics',
					'message_count' => 'num_posts',
				]));
			}

			$importNode->setType($nodeTypeId, $importType);

			$newNodeId = $importNode->save($oldNodeId);
			if ($newNodeId)
			{
				if (!empty($permissionsGrouped[$oldNodeId]))
				{
					$this->setupNodePermissionImport($permissionsGrouped[$oldNodeId], $newNodeId, $node);
				}

				$total++;
				$total += $this->importNodeTree($nodes, $tree, $categoryIdMap, $permissionsGrouped, $oldNodeId);
			}
		}

		return $total;
	}

	protected function setupNodePermissionImport($permissionsByUserGroup, $newNodeId, array $node)
	{
		$this->typeMap('user_group');

		/** @var Permission $permHelper */
		$permHelper = $this->getDataHelper(Permission::class);

		foreach ($permissionsByUserGroup AS $oldGroupId => $perms)
		{
			$newGroupId = $this->lookupId('user_group', $oldGroupId);
			if (!$newGroupId)
			{
				continue;
			}

			$groups = [];
			if (!empty($node['member_groups']))
			{
				$groups = explode(',', $node['member_groups']);
			}

			$newPermissions = $this->calculateForumPermissions($perms, (in_array($oldGroupId, $groups) || in_array(-1, $groups)));

			// now import
			$permHelper->insertContentUserGroupPermissions('node', $newNodeId, $newGroupId, $newPermissions);
		}
	}

	protected function calculateForumPermissions(array $perms, $canView = false)
	{
		$newPerms = [];

		// no equivalents
		$newPerms['general']['viewNode'] = $canView ? 'content_allow' : 'reset';
		$newPerms['forum']['viewContent'] = $canView ? 'content_allow' : 'reset';
		$newPerms['forum']['viewOthers'] = $canView ? 'content_allow' : 'reset';

		$newPerms['forum']['postThread'] = (!empty($perms['post_new']['add_deny']) ? 'content_allow' : 'reset');
		$newPerms['forum']['postReply'] = (!empty($perms['post_reply_own']['add_deny']) ? 'content_allow' : 'reset');
		$newPerms['forum']['editOwnPost'] = (!empty($perms['modify_own']['add_deny']) ? 'content_allow' : 'reset');
		$newPerms['forum']['deleteOwnPost'] = (!empty($perms['delete_replies']['add_deny']) ? 'content_allow' : 'reset');
		$newPerms['forum']['deleteOwnThread'] = (!empty($perms['delete_own']['add_deny']) ? 'content_allow' : 'reset');
		$newPerms['forum']['viewAttachment'] = (!empty($perms['view_attachments']['add_deny']) ? 'content_allow' : 'reset');
		$newPerms['forum']['uploadAttachment'] = (!empty($perms['post_attachment']['add_deny']) ? 'content_allow' : 'reset');
		$newPerms['forum']['votePoll'] = (!empty($perms['poll_vote']['add_deny']) ? 'content_allow' : 'reset');

		return $newPerms;
	}

	// ############################## STEP: MODERATORS #############################

	public function stepModerators(StepState $state, array $stepConfig)
	{
		$this->typeMap('node');

		$moderators = $this->sourceDb->fetchAll("
			SELECT members.id_member, members.id_group, members.additional_groups, mods.*,
				IF (mods.id_member IS NOT NULL, mods.id_member, members.id_member) AS id_member
			FROM members AS members
			LEFT JOIN moderators AS mods ON
				(members.id_member = mods.id_member)
			WHERE mods.id_board IS NOT NULL
				OR (members.id_group = 2 OR FIND_IN_SET(2, members.additional_groups))
		");
		if (!$moderators)
		{
			return $state->complete();
		}

		$modsGrouped = [];
		foreach ($moderators AS $moderator)
		{
			$modsGrouped[$moderator['id_member']][$moderator['id_board']] = $moderator;
		}

		$this->lookup('user', array_keys($modsGrouped));

		/** @var Moderator $modHelper */
		$modHelper = $this->getDataHelper(Moderator::class);

		foreach ($modsGrouped AS $userId => $forums)
		{
			$newUserId = $this->lookupId('user', $userId);
			if (!$newUserId)
			{
				continue;
			}

			$inserted = false;
			$superMod = false;

			foreach ($forums AS $forumId => $moderator)
			{
				$forumPermissions = [];

				if ($forumId)
				{
					$newNodeId = $this->lookupId('node', $forumId);
					if (!$newNodeId)
					{
						continue;
					}

					$forum = $this->sourceDb->fetchRow('
						SELECT id_board, id_profile
						FROM boards
						WHERE id_board = ?
					', $forumId);

					$permSql = $this->sourceDb->query('
						SELECT *
						FROM board_permissions
						WHERE id_profile = ?
					', $forum['id_profile']);
					while ($permission = $permSql->fetch())
					{
						$forumPermissions[$permission['permission']] = $permission;
					}

					$permissions = $this->convertForumPermissionsForUser($forumPermissions);
					$modHelper->importContentModerator($newUserId, 'node', $newNodeId, $permissions);

					$inserted = true;
				}
				else
				{
					$superMod = true;
				}

				if ($inserted || $superMod)
				{
					$modHelper->importModerator($newUserId, $superMod, [User::GROUP_MOD]);

					$state->imported++;
				}
			}
		}

		return $state->complete();
	}

	protected function convertForumPermissionsForUser(array $perms)
	{
		$output = [];

		$output['forum']['editAny'] = (!empty($perms['modify_any']['add_deny']) ? 'content_allow' : '');
		$output['forum']['lockUnlockThread'] = (!empty($perms['lock_any']['add_deny']) ? 'content_allow' : '');
		$output['forum']['stickUnstickThread'] = (!empty($perms['make_sticky']['add_deny']) ? 'content_allow' : '');

		if (!empty($perms['approve_posts']['add_deny']))
		{
			$output['forum']['approveUnapprove'] = 'content_allow';
			$output['forum']['viewModerated'] = 'content_allow';
		}

		if (!empty($perms['remove_any']['add_deny']))
		{
			$output['forum']['deleteAnyPost'] = 'content_allow';
			$output['forum']['deleteAnyThread'] = 'content_allow';
			$output['forum']['viewDeleted'] = 'content_allow';
			$output['forum']['undelete'] = 'content_allow';
		}

		if (!empty($perms['split_any'])
			|| !empty($perms['move_any'])
			|| !empty($perms['merge_any'])
		)
		{
			$output['forum']['manageAnyThread'] = 'content_allow';
		}

		return $output;
	}

	// ############################## STEP: THREADS #############################

	public function getStepEndThreads()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(id_topic)
			FROM topics
		") ?: 0;
	}

	public function stepThreads(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$threads = $this->sourceDb->fetchAllKeyed("
			SELECT topics.*, fp.subject, fp.poster_time,
				lp.poster_time AS last_post_date, lp.poster_name AS last_post_username,
				IF (members.member_name IS NOT NULL, members.member_name, fp.poster_name) AS member_name,
				polls.*
			FROM topics AS topics FORCE INDEX (PRIMARY)
			LEFT JOIN members AS members ON
				(topics.id_member_started = members.id_member)
			INNER JOIN messages AS fp ON
				(topics.id_first_msg = fp.id_msg)
			INNER JOIN messages AS lp ON
				(topics.id_last_msg = lp.id_msg)
			INNER JOIN boards AS boards ON
				(topics.id_board = boards.id_board)
			LEFT JOIN polls AS polls ON
				(topics.id_poll = polls.id_poll)
			WHERE topics.id_topic > ? AND topics.id_topic <= ?
				AND boards.redirect = ''
			ORDER BY topics.id_topic
			LIMIT {$limit}
		", 'id_topic', [$state->startAfter, $state->end]);

		if (!$threads)
		{
			return $state->complete();
		}

		$this->typeMap('node');
		$this->lookup('user', $this->pluck($threads, 'id_member_started'));

		foreach ($threads AS $oldThreadId => $thread)
		{
			$state->startAfter = $oldThreadId;

			$nodeId = $this->lookupId('node', $thread['id_board']);

			if (!$nodeId)
			{
				continue;
			}

			/** @var Thread $import */
			$import = $this->newHandler(Thread::class);

			$import->bulkSet($this->mapXfKeys($thread, [
				'reply_count' => 'num_replies',
				'view_count' => 'num_views',
				'sticky' => 'is_sticky',
				'last_post_date',
				'post_date' => 'poster_time',
			]));

			$import->bulkSet([
				'discussion_open' => ($thread['locked'] == 0 ? 1 : 0),
				'node_id' => $nodeId,
				'user_id' => $this->lookupId('user', $thread['id_member_started'], 0),
				'discussion_state' => $thread['approved'] ? 'visible' : 'moderated',
			]);

			$import->set('title', $thread['subject'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$import->set('username', $thread['member_name'], [EntityEmulator::UNHTML_ENTITIES => true]);
			$import->set('last_post_username', $thread['last_post_username'], [EntityEmulator::UNHTML_ENTITIES => true]);

			$subs = $this->sourceDb->fetchAllColumn("
				SELECT id_member
				FROM log_notify
				WHERE id_topic = {$oldThreadId}
			");

			if ($subs)
			{
				$this->lookup('user', $subs);

				foreach ($subs AS $userId)
				{
					$newUserId = $this->lookupId('user', $userId);
					if (!$newUserId)
					{
						continue;
					}

					$import->addThreadWatcher($newUserId, true);
				}
			}

			if ($thread['id_poll'])
			{
				/** @var Poll $importPoll */
				$importPoll = $this->newHandler(Poll::class);
				$importPoll->question = $this->convertContentToBbCode($thread['question']);
				$importPoll->bulkSet([
					'public_votes' => $thread['hide_results'],
					'max_votes' => $thread['max_votes'] ? 1 : 0,
					'change_vote' => $thread['change_vote'],
					'close_date' => $thread['expire_time'],
				]);

				$responses = $this->sourceDb->fetchPairs('
					SELECT id_choice, label
					FROM poll_choices
					WHERE id_poll = ?
				', $thread['id_poll']);
				if (!$responses)
				{
					continue;
				}

				foreach ($responses AS &$value)
				{
					$value = $this->convertContentToBbCode($value);
				}

				$importResponses = [];

				foreach ($responses AS $choiceId => $label)
				{
					/** @var PollResponse $importResponse */
					$importResponse = $this->newHandler(PollResponse::class);
					$importResponse->preventRetainIds();
					$importResponse->response = $label;

					$importResponses[$choiceId] = $importResponse;

					$importPoll->addResponse($choiceId, $importResponse);
				}

				$votes = $this->sourceDb->fetchAll('
					SELECT id_member, id_choice
					FROM log_polls
					WHERE id_poll = ?
				', $thread['id_poll']);

				$this->lookup('user', $this->pluck($votes, 'id_member'));

				foreach ($votes AS $vote)
				{
					$voteUserId = $this->lookupId('user', $vote['id_member']);
					if (!$voteUserId)
					{
						continue;
					}

					$voteOption = $vote['id_choice'];

					if (!array_key_exists($voteOption, $importResponses))
					{
						continue;
					}

					$importResponses[$voteOption]->addVote($voteUserId);
				}

				$import->addPoll($thread['id_poll'], $importPoll);
			}

			if ($newThreadId = $import->save($oldThreadId))
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ############################## STEP: POSTS #############################

	protected function getThreadIdsForPostsStep($startAfter, $end, $threadLimit)
	{
		return $this->sourceDb->fetchAllColumn("
			SELECT id_topic
			FROM topics
			WHERE id_topic > ? AND id_topic <= ?
			ORDER BY id_topic
			LIMIT {$threadLimit}
		", [$startAfter, $end]);
	}

	protected function getPostsForPostsStep($threadId, $startDate)
	{
		$limit = self::$postsStepLimit;

		return $this->sourceDb->fetchAll("
			SELECT messages.*,
				IF(members.member_name IS NOT NULL, members.member_name, messages.poster_name) AS member_name
			FROM messages AS messages
			LEFT JOIN members AS members ON (messages.id_member = members.id_member)
			WHERE messages.id_topic = ?
				AND messages.poster_time > ?
			ORDER BY messages.poster_time
			LIMIT {$limit}
		", [$threadId, $startDate]);
	}

	protected function lookupUsers(array $posts)
	{
		$this->lookup('user', $this->pluck($posts, 'id_member'));
	}

	protected function getPostDateField()
	{
		return 'poster_time';
	}

	protected function getPostIdField()
	{
		return 'id_msg';
	}

	protected function handlePostImport(array $post, $newThreadId, StepState $state)
	{
		/** @var Post $import */
		$import = $this->newHandler(Post::class);

		$import->bulkSet([
			'thread_id' => $newThreadId,
			'user_id' => $this->lookupId('user', $post['id_member']),
			'post_date' => $post['poster_time'],
			'message' => $this->convertContentToBbCode($post['body']),
			'message_state' => $post['approved'] ? 'visible' : 'moderated',
			'position' => $state->extra['postPosition'],
		]);

		$import->set('username', $post['member_name'], [EntityEmulator::UNHTML_ENTITIES => true]);

		$import->setLoggedIp($post['poster_ip']);

		return $import;
	}

	// ########################### STEP: ATTACHMENTS ###############################

	public function getStepEndAttachments()
	{
		return $this->sourceDb->fetchOne("
			SELECT MAX(id_attach)
			FROM attachments
		") ?: 0;
	}

	public function stepAttachments(StepState $state, array $stepConfig, $maxTime, $limit = 1000)
	{
		$timer = new Timer($maxTime);

		$attachments = $this->sourceDb->fetchAllKeyed("
			SELECT attachments.*, messages.id_member, messages.poster_time
			FROM attachments AS attachments
			INNER JOIN messages AS messages ON
			    (messages.id_msg = attachments.id_msg)
			WHERE attachments.id_attach > ? AND attachments.id_attach <= ?
				AND attachments.id_msg > 0
				AND attachments.id_member = 0
				AND attachments.attachment_type = 0
			ORDER BY attachments.id_attach
			LIMIT {$limit}
		", 'id_attach', [$state->startAfter, $state->end]);

		if (!$attachments)
		{
			return $state->complete();
		}

		$this->lookup('user', $this->pluck($attachments, 'id_member'));
		$this->lookup('post', $this->pluck($attachments, 'id_msg'));

		foreach ($attachments AS $oldId => $attachment)
		{
			$state->startAfter = $oldId;

			$newPostId = $this->lookupId('post', $attachment['id_msg']);
			if (!$newPostId)
			{
				continue;
			}

			$attachPath = $this->getAttachmentFilename($attachment);
			if (!file_exists($attachPath))
			{
				continue;
			}

			/** @var Attachment $import */
			$import = $this->newHandler(Attachment::class);
			$import->bulkSet([
				'content_type' => 'post',
				'content_id'   => $newPostId,
				'attach_date'  => $attachment['poster_time'],
				'view_count'   => $attachment['downloads'],
				'unassociated' => false,
			]);
			$import->setDataExtra('upload_date', $attachment['poster_time']);

			$import->setDataUserId($this->lookupId('user', $attachment['id_member']));
			$import->setSourceFile($attachPath, $this->convertToUtf8($attachment['filename'], null, true));
			$import->setContainerCallback([$this, 'rewriteEmbeddedAttachments']);

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		File::cleanUpTempFiles();

		return $state->resumeIfNeeded();
	}

	protected function convertContentToBbCode($string, $strip = false)
	{
		$string = $this->convertToUtf8($string, null, true);

		// Handles <br /> in message content
		$string = preg_replace('#<br\s*/?>#i', "\n", $string);

		// Handles quotes
		$string = preg_replace('#\[quote\sauthor=(.+)\slink=[^\]]+]#siU', '[quote="$1"]', $string);

		// Handles images
		$string = preg_replace('#\[img.*](.*)\[/img\]#siU', '[IMG]$1[/IMG]', $string);

		// Handles sizes
		$string = preg_replace_callback(
			'#\[size=([^\]]+)(pt|px|em|)\](.*)\[/size\]#siU',
			function (array $match)
			{
				$size = intval($match[1]);
				$unit = $match[2];
				$text = $match[3];

				// Converts pt or em sizes to px (approximately)...
				switch ($unit)
				{
					case 'pt':
						$size = round($size * 1.333333);
						$unit = 'px';
						break;

					case 'em':
						$size = round($size * 15);
						$unit = 'px';
						break;
				}

				return '[SIZE=' . strval($size) . $unit . ']' . $text . '[/SIZE]';
			},
			$string
		);

		// Handles list items
		$string = str_ireplace('[li]', '[*]', $string);
		$string = str_ireplace('[/li]', '', $string);

		// Handles FTP tags (converts to URLs)
		$string = str_ireplace('[ftp', '[URL', $string);
		$string = str_ireplace('[/ftp]', '[/URL]', $string);

		// Handles IRUL tags (converts to URLs)
		$string = str_ireplace('[iurl', '[URL', $string);
		$string = str_ireplace('[/iurl]', '[/URL]', $string);

		// Handles non-breaking spaces
		$string = str_ireplace('&nbsp;', ' ', $string);

		// close enough to code
		$string = preg_replace('#\[(pre|tt)\](.*)\[/(pre|tt)\]#siU', '[CODE]$2[/CODE]', $string);

		// more or less a URL
		$string = preg_replace('#\[ftp\](.*)\[/ftp\]#siU', '[URL]$1[/URL]', $string);

		// Attachments
		$string = preg_replace('#\[attach=(\d+)\]#', '[ATTACH]$1[/ATTACH]', $string);

		// no equivalents, strip
		$string = preg_replace('#\[hr\]#siU', '', $string);
		$string = preg_replace('#\[move\](.*)\[/move\]#siU', '$1', $string);
		$string = preg_replace('#\[sup\](.*)\[/sup\]#siU', '$1', $string);
		$string = preg_replace('#\[sub\](.*)\[/sub\]#siU', '$1', $string);
		$string = preg_replace('#\[glow=([^\]]+)\](.*)\[/glow\]#siU', '$2', $string);
		$string = preg_replace('#\[shadow=([^\]]+)\](.*)\[/shadow\]#siU', '$2', $string);

		$string = preg_replace(
			'#\[([a-z0-9_\*]+(="[^"]*"|=[^\]]*)?)\]#siU',
			($strip ? '' : '[$1]'),
			$string
		);
		$string = preg_replace(
			'#\[/([a-z0-9_\*]+)(:[a-z])?\]#siU',
			($strip ? '' : '[/$1]'),
			$string
		);

		return $string;
	}

	protected function getAttachmentFilename(array $attachment)
	{
		switch ($attachment['attachment_type'])
		{
			case 0: // attachment in attachment directory
				$attachmentConfig = $this->getStepSpecificConfig('attachments', $this->session->stepConfig);
				$attachmentPaths = $attachmentConfig['attach_paths'];

				if (!isset($attachmentPaths[$attachment['id_folder']]))
				{
					return null;
				}

				$attachmentPath = $attachmentPaths[$attachment['id_folder']];

				if (!empty($attachment['file_hash']))
				{
					$path = "{$attachmentPath}/{$attachment['id_attach']}_{$attachment['file_hash']}";
					if (file_exists($path))
					{
						// legacy naming convention
						return $path;
					}
					else
					{
						// new naming convention
						return "$path.dat";
					}
				}
				else // legacy file name handling
				{
					$filename = $attachment['filename'];

					$cleanName = preg_replace(['/\s/', '/[^\w_\.\-]/'], ['_', ''], $filename);
					$encName = $attachment['id_attach'] . '_' . strtr($cleanName, '.', '_') . md5($cleanName);
					$cleanName = preg_replace('~\.[\.]+~', '.', $cleanName);

					$encPath = "{$attachmentPath}/$encName";
					$cleanPath = "{$attachmentPath}/$cleanName";

					if (file_exists($encPath))
					{
						return $encPath;
					}
					else
					{
						return $cleanPath;
					}
				}

				// no break
			case 1: // avatar in custom directory
				$userConfig = $this->getStepSpecificConfig('users', $this->session->stepConfig);
				$avatarPath = $userConfig['avatar_path'];

				return "{$avatarPath}/{$attachment['filename']}";
		}

		return null;
	}
}
