<?php

namespace XFI\Import\Importer;

use XF\Db\AbstractAdapter;
use XF\Import\Data\Reaction;
use XF\Import\StepState;

use function intval;

class XenForo21 extends XenForo2
{
	public static function getListInfo()
	{
		return [
			'target' => 'XenForo',
			'source' => 'XenForo 2.1',
		];
	}

	protected function validateVersion(AbstractAdapter $db, &$error)
	{
		$versionId = $db->fetchOne("SELECT option_value FROM xf_option WHERE option_id = 'currentVersionId'");
		if (!$versionId || intval($versionId) < 2010031 || intval($versionId) >= 2020031)
		{
			$error = \XF::phrase('xfi_you_may_only_import_from_xenforo_x', ['version' => '2.1']);
			return false;
		}

		return true;
	}

	public function getSteps()
	{
		$steps = parent::getSteps();

		$steps = $this->extendSteps($steps, [
			'title' => \XF::phrase('reaction_definitions'),
			'depends' => ['posts'],
		], 'reactions', 'likes');

		$steps = $this->extendSteps($steps, [
			'title' => \XF::phrase('reaction_content'),
			'depends' => ['reactions'],
		], 'reactionContent', 'reactions');

		$steps['bookmarks'] = [
			'title' => \XF::phrase('bookmarks'),
			'depends' => ['users'],
		];

		unset($steps['likes']);

		return $steps;
	}

	protected function getBasicThreadMapping()
	{
		return [
			'title',
			'reply_count',
			'view_count',
			'username',
			'post_date',
			'sticky',
			'discussion_state',
			'discussion_open',
			'discussion_type',
			'first_post_reaction_score',
			'last_post_date',
			'last_post_username',
		];
	}

	protected function setupThreadImport(array $thread, $nodeId)
	{
		$import = parent::setupThreadImport($thread, $nodeId);

		$reactions = $this->decodeValue($thread['first_post_reactions'], 'json-array');
		if ($reactions)
		{
			$import->first_post_reactions = $reactions;
		}

		return $import;
	}

	// ########################### STEP: REACTIONS ###############################

	public function stepReactions(StepState $state)
	{
		$reactions = $this->sourceDb->fetchAll("
			SELECT reaction.*, ptitle.phrase_text AS title
			FROM xf_reaction AS reaction
			INNER JOIN xf_phrase AS ptitle ON
				(ptitle.language_id = 0 AND ptitle.title = CONCAT('reaction_title.', reaction.reaction_id))
			ORDER BY reaction.reaction_id
		");

		foreach ($reactions AS $reaction)
		{
			$oldId = $reaction['reaction_id'];

			if ($oldId <= 6)
			{
				// Reactions 1 to 6 are most likely going to be the default Like, Love, Haha, Sad, Angry, Wow reactions
				$this->logHandler(Reaction::class, $oldId, $oldId);
			}
			else
			{
				$data = $this->mapKeys($reaction, [
					'text_color',
					'image_url',
					'image_url_2x',
					'sprite_mode',
					'reaction_score',
					'display_order',
					'active',
				]);

				/** @var Reaction $import */
				$import = $this->newHandler(Reaction::class);
				$import->sprite_params = @json_decode($reaction['sprite_params'], true) ?: [];
				$import->setTitle($reaction['title']);
				$import->bulkSet($data);
				$import->save($oldId);
			}

			$state->imported++;
		}

		return $state->complete();
	}

	// ########################### STEP: REACTION CONTENT ###############################

	public function getStepEndReactionContent()
	{
		return $this->getMaxReactionContentIdForContentTypes(
			['conversation_message', 'post', 'profile_post', 'profile_post_comment']
		);
	}

	public function stepReactionContent(StepState $state, array $stepConfig, $maxTime)
	{
		return $this->getReactionContentStepStateForContentTypes(
			['conversation_message', 'post', 'profile_post', 'profile_post_comment'],
			$state,
			$stepConfig,
			$maxTime
		);
	}

	// ########################### STEP: BOOKMARKS ###############################

	public function getStepEndBookmarks()
	{
		return $this->getMaxBookmarkIdForContentTypes(['node', 'post']);
	}

	public function stepBookmarks(StepState $state, array $stepConfig, $maxTime)
	{
		return $this->getBookmarksStepStateForContentTypes(
			['node', 'post'],
			$state,
			$stepConfig,
			$maxTime
		);
	}
}
