<?php

namespace XFI\Import\Importer;

use XF\Db\AbstractAdapter;
use XF\Entity\UserProfile;
use XF\Import\Data\UsernameChange;
use XF\Import\DataHelper\ProfileBanner;
use XF\Import\DataHelper\Thread;
use XF\Import\StepState;
use XF\Timer;

use function intval;

class XenForo22 extends XenForo21
{
	public static function getListInfo()
	{
		return [
			'target' => 'XenForo',
			'source' => 'XenForo 2.2',
		];
	}

	protected function validateVersion(AbstractAdapter $db, &$error)
	{
		$versionId = $db->fetchOne("SELECT option_value FROM xf_option WHERE option_id = 'currentVersionId'");
		if (!$versionId || intval($versionId) < 2020031 || intval($versionId) >= 2030031)
		{
			$error = \XF::phrase('xfi_you_may_only_import_from_xenforo_x', ['version' => '2.2']);
			return false;
		}

		return true;
	}

	public function getSteps()
	{
		$steps = parent::getSteps();

		$steps = $this->extendSteps($steps, [
			'title' => \XF::phrase('username_changes'),
			'depends' => ['users'],
		], 'usernameChanges', 'followIgnore');

		$steps = $this->extendSteps($steps, [
			'title' => \XF::phrase('xfi_profile_banners'),
			'depends' => ['users'],
		], 'profileBanners', 'avatars');

		$steps = $this->extendSteps($steps, [
			'title' => \XF::phrase('xfi_content_votes'),
			'depends' => ['posts'],
		], 'contentVotes', 'attachments');

		$steps = $this->extendSteps($steps, [
			'title' => \XF::phrase('xfi_question_solutions'),
			'depends' => ['posts'],
		], 'questionSolutions', 'posts');

		return $steps;
	}

	// ########################### STEP: USERS ###############################

	protected function setupImportUser(array $user)
	{
		$import = parent::setupImportUser($user);

		$userData = $this->mapKeys($user, [
			'username_date',
			'security_lock',
			'last_summary_email_date',
		]);
		$import->bulkSetDirect('user', $userData);

		return $import;
	}

	// ########################### STEP: USERNAME CHANGES ###############################

	public function getStepEndUsernameChanges()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(change_id) FROM xf_username_change") ?: 0;
	}

	public function stepUsernameChanges(StepState $state, array $stepConfig, $maxTime)
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$changes = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM xf_username_change
			WHERE change_id > ? AND change_id <= ?
			ORDER BY change_id
			LIMIT {$limit}
		", 'change_id', [$state->startAfter, $state->end]);
		if (!$changes)
		{
			return $state->complete();
		}

		$mapUserIds = [];
		foreach ($changes AS $change)
		{
			$mapUserIds[] = $change['user_id'];
			if ($change['change_user_id'])
			{
				$mapUserIds[] = $change['change_user_id'];
			}
			if ($change['moderator_user_id'])
			{
				$mapUserIds[] = $change['moderator_user_id'];
			}
		}

		$this->lookup('user', $mapUserIds);

		foreach ($changes AS $change)
		{
			$oldId = $change['change_id'];
			$state->startAfter = $oldId;

			$userId = $this->lookupId('user', $change['user_id']);
			if (!$userId)
			{
				continue;
			}

			/** @var UsernameChange $import */
			$import = $this->newHandler(UsernameChange::class);
			$import->bulkSet($this->mapKeys($change, [
				'old_username',
				'new_username',
				'change_reason',
				'change_state',
				'change_date',
				'reject_reason',
				'visible',
			]));
			$import->user_id = $userId;
			$import->change_user_id = $change['change_user_id']
				? $this->lookupId('user', $change['change_user_id'], 0)
				: 0;
			$import->moderator_user_id = $change['moderator_user_id']
				? $this->lookupId('user', $change['moderator_user_id'], 0)
				: 0;

			$newId = $import->save($oldId);
			if ($newId)
			{
				$state->imported++;
			}

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: PROFILE BANNERS ###############################

	public function getStepEndProfileBanners()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(user_id) FROM xf_user_profile WHERE banner_date > 0") ?: 0;
	}

	public function stepProfileBanners(StepState $state, array $stepConfig, $maxTime)
	{
		$limit = 500;
		$timer = new Timer($maxTime);

		$users = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM xf_user_profile
			WHERE user_id > ? AND user_id <= ? AND banner_date > 0
			ORDER BY user_id
			LIMIT {$limit}
		", 'user_id', [$state->startAfter, $state->end]);
		if (!$users)
		{
			return $state->complete();
		}

		$requiredKeys = array_keys($this->app->container('profileBannerSizeMap'));

		/** @var ProfileBanner $bannerHelper */
		$bannerHelper = $this->getDataHelper(ProfileBanner::class);

		$mappedUserIds = $this->lookup('user', array_keys($users));

		foreach ($users AS $user)
		{
			$oldId = $user['user_id'];
			$state->startAfter = $oldId;

			$mappedUserId = $mappedUserIds[$oldId];
			if (!$mappedUserId)
			{
				continue;
			}

			$baseSourceFile = sprintf(
				'%s/profile_banners/{size}/%d/%d.jpg',
				$this->baseConfig['data_dir'],
				floor($oldId / 1000),
				$oldId
			);
			$sourceFiles = [];
			foreach ($requiredKeys AS $size)
			{
				$sourceFile = str_replace('{size}', $size, $baseSourceFile);
				if (file_exists($sourceFile) && is_readable($sourceFile))
				{
					$sourceFiles[$size] = str_replace('{size}', $size, $sourceFile);
				}
			}

			$isValid = true;
			foreach ($requiredKeys AS $sizeKey)
			{
				if (!isset($sourceFiles[$sizeKey]))
				{
					$isValid = false;
					break;
				}
			}

			if (!$isValid)
			{
				continue;
			}

			/** @var UserProfile|null $targetProfile */
			$targetProfile = $this->em()->find(UserProfile::class, $mappedUserId);
			if (!$targetProfile)
			{
				continue;
			}

			if (!$bannerHelper->copyFinalBannerFiles($sourceFiles, $targetProfile))
			{
				continue;
			}

			$targetProfile->fastUpdate([
				'banner_date' => $user['banner_date'],
				'banner_position_y' => $user['banner_position_y'],
			]);

			$state->imported++;

			$this->em()->detachEntity($targetProfile);

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: NODES ###############################

	protected function setupNodeForumImport(array $data)
	{
		$handler = parent::setupNodeForumImport($data);

		$handler->allow_index = $data['allow_index'];
		$handler->index_criteria = $this->decodeValue($data['index_criteria'], 'json-array');

		$handler->forum_type_id = $data['forum_type_id'];
		$handler->type_config = $this->decodeValue($data['type_config'], 'json-array');

		return $handler;
	}

	// ########################### STEP: THREADS ###############################

	protected function setupThreadImport(array $thread, $nodeId)
	{
		$handler = parent::setupThreadImport($thread, $nodeId);

		$handler->type_data = $this->decodeValue($thread['type_data'], 'json-array');

		return $handler;
	}

	// ########################### STEP: POSTS ###############################

	protected function setupImportPost(array $post)
	{
		$import = parent::setupImportPost($post);
		if (!$import)
		{
			return $import;
		}

		$import->type_data = $this->decodeValue($post['type_data'], 'json-array');

		return $import;
	}

	// ########################### STEP: QUESTION SOLUTIONS ###############################

	public function getStepEndQuestionSolutions()
	{
		return $this->sourceDb->fetchOne("SELECT MAX(thread_id) FROM xf_thread_question WHERE solution_post_id > 0") ?: 0;
	}

	public function stepQuestionSolutions(StepState $state, array $stepConfig, $maxTime)
	{
		$limit = 1000;
		$timer = new Timer($maxTime);

		$questions = $this->sourceDb->fetchAllKeyed("
			SELECT *
			FROM xf_thread_question
			WHERE thread_id > ? AND thread_id <= ? AND solution_post_id > 0
			ORDER BY thread_id
			LIMIT {$limit}
		", 'thread_id', [$state->startAfter, $state->end]);
		if (!$questions)
		{
			return $state->complete();
		}

		$mappedThreadIds = $this->lookup('thread', array_keys($questions));

		$mapPostIds = [];
		foreach ($questions AS $question)
		{
			$mapPostIds[] = $question['solution_post_id'];
		}

		$this->lookup('post', $mapPostIds);

		/** @var Thread $threadHelper */
		$threadHelper = $this->getDataHelper(Thread::class);

		foreach ($questions AS $question)
		{
			$oldId = $question['thread_id'];
			$state->startAfter = $oldId;

			$threadId = $mappedThreadIds[$oldId];
			if (!$threadId)
			{
				continue;
			}

			$postId = $this->lookupId('post', $question['solution_post_id']);
			if (!$postId)
			{
				continue;
			}

			$threadHelper->updateQuestionSolution($threadId, $postId);

			$state->imported++;

			if ($timer->limitExceeded())
			{
				break;
			}
		}

		return $state->resumeIfNeeded();
	}

	// ########################### STEP: CONTENT VOTES ###############################

	public function getStepEndContentVotes()
	{
		return $this->getMaxContentVoteIdForContentTypes(
			['thread', 'post']
		);
	}

	public function stepContentVotes(StepState $state, array $stepConfig, $maxTime)
	{
		return $this->getContentVoteStepStateForContentTypes(
			['thread', 'post'],
			$state,
			$stepConfig,
			$maxTime
		);
	}
}
